#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2024. All rights reserved.
import argparse
import json
import os
import stat


def load_file(file_path: str):
    found = False

    def object_hook(ordered_pairs):
        nonlocal found
        result_dict = {'__duplicate_keys': []}
        for key, value in ordered_pairs:
            if key in result_dict:
                found = True
                result_dict['__duplicate_keys'].append(key)
            else:
                result_dict[key] = value
        return result_dict
    with open(file_path, "r", encoding="utf-8") as file_descriptor:
        data = json.load(file_descriptor, object_pairs_hook=object_hook)
    return found, data


class DuplicateKeyChecker:
    def __init__(self, args):
        self.input_file = args.input
        self.output_dir = args.output
        if self.input_file is None or not os.path.isfile(self.input_file):
            raise RuntimeError("缺少参数或输入文件不存在")
        if self.output_dir is None or not os.path.isdir(self.output_dir):
            raise RuntimeError("缺少参数或输出目录不存在")

    def run(self):
        with open(self.input_file, "r", encoding="utf-8") as file_descriptor:
            files_to_check = json.load(file_descriptor)
        index = 1
        for file_path in files_to_check:
            if not os.path.isfile(file_path):
                continue
            found, data = load_file(file_path)
            if found:
                self.write_output(index, {"filePath": file_path, "fileContent": data})
                index += 1

    def write_output(self, index, output_data):
        output_file = os.path.join(self.output_dir, f"{index}.json")
        if os.path.islink(output_file):
            raise RuntimeError("输出文件路径是软链接")
        with os.fdopen(os.open(output_file, os.O_WRONLY | os.O_CREAT | os.O_TRUNC,
                               stat.S_IWUSR | stat.S_IRUSR), 'w') as file_descriptor:
            json.dump(output_data, file_descriptor, ensure_ascii=False)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="json对象key重复检查")
    parser.add_argument("-i", "--input", help="输入文件")
    parser.add_argument("-o", "--output", help="输出目录")
    args, _ = parser.parse_known_args()
    DuplicateKeyChecker(args).run()