//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
import { traduction } from '@/utils/language';
// 格式错误
const formatErrMessage = traduction('VALIDTOR_FORMAT_ERROR');

interface IValidatorRule {
  field: string;
  fullFiled: string;
  message: string;
  type: 'string';
  [propName: string]: any;
  validator: () => void;
}

interface IParams {
  pattern?: RegExp;
  message?: string;
  trigger?: string;
  range?: number[];
  [propName: string]: any;
}

export class BaseValidateRuleItems {
  constructor() {}

  // 是否必传
  public static isRequired = (param?: IParams) => {
    const rule = {
      required: false,
      type: 'string',
      message: formatErrMessage,
      trigger: 'change',
      transform: (value: string | number) => {
        return value?.toString();
      },
    };

    if (param) {
      Object.assign(rule, param);
    }
    return rule;
  };

  // 数字类型
  public static isNumber = (param?: IParams) => {
    const rule = {
      type: 'number',
      message: formatErrMessage,
      trigger: 'change',
      transform: (value: string) => {
        return Number(value);
      },
    };

    if (param) {
      Object.assign(rule, param);
    }
    return rule;
  };

  // 整数类型
  public static isInteger = (param?: IParams) => {
    const rule = {
      type: 'integer',
      message: formatErrMessage,
      trigger: 'change',
      transform: (value: string) => {
        return Number(value);
      },
      validator: (rules: IValidatorRule, value: any, callback: (error?: Error) => void): void => {
        if (value % 1 === 0) {
          callback();
        } else {
          callback(new Error(rules.message));
        }
      },
    };

    if (param) {
      Object.assign(rule, param);
    }
    return rule;
  };

  // 在两个数区间内
  public static inRange = (param?: IParams) => {
    const range = param?.range || [0, 255];
    const rule = {
      type: 'number',
      message: traduction('VALID_VALUE_RANGE_TIP', range),
      trigger: 'change',
      min: range[0],
      max: range[1],
      transform: (value: string | number) => {
        if (value || value === 0) {
          return Number(value);
        } else {
          return '';
        }
      },
      validator: BaseValidateRuleItems.inRangeValidator,
    };

    if (param) {
      Object.assign(rule, param);
    }
    return rule;
  };

  public static inRangeValidator = (rule: IValidatorRule, value: any, callback: (error?: Error) => void) => {
    const { min, max } = rule;
    if (value || value === 0) {
      if ((min === 0 || min) && max && min <= max && min <= value && max >= value) {
        callback();
      } else {
        callback(new Error(rule.message));
      }
    } else {
      callback();
    }
  };

  // 字符串长度在某个区间
  public static lengthRange = (param?: IParams) => {
    const lengthRange = param?.lengthRange || [0, 255];
    const rule = {
      type: 'string',
      message: traduction('VALIDATOR_PWD_STANDARD', lengthRange),
      trigger: 'change',
      minLength: lengthRange[0],
      maxLength: lengthRange[1],
      transform: (value: any) => {
        return String(value);
      },
      validator: (rules: IValidatorRule, value: any, callback: (error?: Error) => void): void => {
        const { minLength, maxLength } = rules;
        const len = value.length;
        if (value && minLength >= 0 && maxLength && minLength <= maxLength && minLength <= len && maxLength >= len) {
          callback();
        } else {
          callback(new Error(rules.message));
        }
      },
    };

    if (param) {
      Object.assign(rule, param);
    }
    return rule;
  };

  // 正则表达式校验
  public static pattern = (param?: IParams) => {
    const rule = {
      type: 'string',
      trigger: 'change',
      message: traduction('VALIDATOR_INCORRECT_REINPUT'),
      validator: (rules: IValidatorRule, value: any, callback: (error?: Error) => void): void => {
        if (param?.pattern?.test(value)) {
          callback();
        } else {
          callback(new Error(rules.message));
        }
      },
    };

    if (param) {
      Object.assign(rule, param);
    }
    return rule;
  };
}
