//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
import Color from '@/components/Customize/color';
import { setColors } from '@/model/colors';
import { setStoreData } from '@/utils/composition';
import { ColorTheme } from './../../model/base-enum';
import {
  IAlarmColor,
  IColorConfig,
  ILayout,
  IStateColor,
  IStyleConfig,
} from './styleConfig.datatype';
import { setCircles } from '@/model/echartsCircles';
export default class CustomizeService {
  public bodyStyles: any;
  constructor() {
    this.bodyStyles = document.body.style;
  }

  // 设置css变量 (e.g. key '--o-color-primary' value '#fff' priority 'important')
  public setVar(key: string, value: string, priority?: string) {
    this.bodyStyles.setProperty(key, value, priority || '');
  }

  // 设置颜色主题，light/dark
  public setColorTheme(theme: ColorTheme) {
    document.body.setAttribute('theme', theme);
    // 重置echarts图表主题色
    setColors();
    // 根据主题设置图表的使用图片
    setCircles();
    // 用于图表监听然后根据新的主题进行重新setOption
    setStoreData('loct', 'systemTheme', theme);
  }
  
  // 初始化定制,webStyleConfig：白牌包定制配置，可为null，customConfig：本地定制配置。
  public initCustomize(webStyleConfig: any, customConfig: any, isCustomizeToolConfig?: boolean) {
    /* 使用本地定制文件场景如下：
     *     1、四种风格定制场景 (e.g 不需要定制首页的卡片位置和大小)
     *     2、本地定制文件优先级高
     *     3、无白牌定制文件
     */
    if (
      customConfig &&
      (!webStyleConfig ||
        webStyleConfig.pageStyle ||
        (customConfig.priority && customConfig.priority > 0))
    ) {
      this.setConfig(
        customConfig,
        !(webStyleConfig?.pageStyle !== 'default'),
        isCustomizeToolConfig,
      );
    } else {
      this.setConfig(webStyleConfig, true, isCustomizeToolConfig);
    }
  }

  private setConfig(config: any, isNeedSetInfo: boolean, isCustomizeToolConfig?: boolean) {
    if (config.color) {
      this.initColorCustomize(config.color);
    }

    if (config.layout) {
      this.initLayout(config.layout);
    }

    if (config.info && isNeedSetInfo && !isCustomizeToolConfig) {
      localStorage.setItem('ptc', JSON.stringify(config.info));
    }
  }

  private initLayout(layout: ILayout) {
    const customizeStyle = JSON.parse(localStorage.getItem('customizeStyle') || '{}');
    customizeStyle.layout = layout;
    localStorage.setItem('customizeStyle', JSON.stringify(customizeStyle));
    if (layout.control) {
      this.setRadius(layout.control.radius);
      this.setShadow(layout.control.shadow);
    }
  }

  // 初始化颜色定制化
  public initColorCustomize(colorConfig: IColorConfig) {
    // 计算hover样式用的混合色，dark模式：黑色，light:#fff
    let mixColor = '#fff';
    // theme要先于其他配置，便于设置混合色等基础色调
    if (colorConfig.themeMode) {
      this.setColorTheme(ColorTheme[colorConfig.themeMode]);
      if (ColorTheme[colorConfig.themeMode] === 'dark') {
        mixColor = '#000';
        // 状态色和品牌色区分开，深色模式下填充色赋值
        this.setVar('--o-color-state-primary-secondary', '#052D5A');
        // 深色模式下，登录框背景色
        this.setVar('--o-color-login-card-primary', '#3E4551');
      }
    }

    Object.keys(colorConfig).forEach(key => {
      if (!colorConfig[key] || colorConfig[key] === '') {
        return;
      }
      switch (key) {
        case 'brandColor': {
          this.setVar('--o-color-primary', colorConfig[key]);
          // hover色,调和计算（mix mixColor 32%）
          let color = new Color(colorConfig[key]);
          this.setVar('--o-color-primary-secondary', color.mixin(mixColor, 0.68).value);
          // disabled色,调和计算（mix mixColor 72%）
          this.setVar('--o-color-primary-fourth', color.mixin(mixColor, 0.28).value);
          // video存在录像的卡片颜色,透明度为0.7（B3)
          this.setVar('--o-color-video-primary', `${colorConfig[key]}B3`);
          // 定制工具入口icon的阴影色（mix 黑色 48%）,透明度为0.4(66)
          this.setVar('--o-color-customize-shadow', `${color.mixin('#000', 0.52).value}66`);
          break;
        }
        case 'background':
          this.setVar('--o-bg-color-dark', colorConfig[key][colorConfig.themeMode]);
          break;
        case 'dividerColor':
          this.setVar('--o-border-color-light', colorConfig[key]);
          break;
        case 'alarmColor':
          this.initAlarmColor(colorConfig[key], mixColor);
          break;
        case 'stateColor':
          this.initStateColor(colorConfig[key], mixColor);
          break;
        default:
          break;
      }
    });
  }

  private initAlarmColor(alarmConfig: IAlarmColor, mixColor: string) {
    Object.keys(alarmConfig).forEach(alarmKey => {
      if (!alarmConfig[alarmKey] || alarmConfig[alarmKey] === '') {
        return;
      }
      const color = new Color(mixColor);
      switch (alarmKey) {
        case 'criticalColor':
          this.setVar('--o-color-danger', alarmConfig[alarmKey]);
          this.setVar('--o-color-danger-secondary', color.mixin(alarmConfig[alarmKey], 0.72).value);
          this.setVar('--o-color-danger-tertiary', color.mixin(alarmConfig[alarmKey], 0.85).value);
          // 告警&事件页面紧急提示框背景色，透明度15%(26)
          this.setVar('--o-color-danger-background', `${alarmConfig[alarmKey]}26`);
          break;
        case 'majorColor':
          this.setVar('--o-color-major', alarmConfig[alarmKey]);
          this.setVar('--o-color-major-secondary', color.mixin(alarmConfig[alarmKey], 0.72).value);
          this.setVar('--o-color-major-tertiary', color.mixin(alarmConfig[alarmKey], 0.85).value);
          // 告警&事件页面严重提示框背景色，透明度15%(26)
          this.setVar('--o-color-major-background', `${alarmConfig[alarmKey]}26`);
          break;
        case 'warningColor':
          this.setVar('--o-color-warning', alarmConfig[alarmKey]);
          this.setVar(
            '--o-color-warning-secondary',
            color.mixin(alarmConfig[alarmKey], 0.72).value,
          );
          // 告警&事件页面轻微提示框背景色，透明度15%(26)
          this.setVar('--o-color-warning-background', `${alarmConfig[alarmKey]}26`);
          break;
        default:
          break;
      }
    });
  }

  private initStateColor(stateConfig: IStateColor, mixColor: string) {
    let color;
    let contentColor;
    Object.keys(stateConfig).forEach(stateKey => {
      if (!stateConfig[stateKey] || stateConfig[stateKey] === '') {
        return;
      }
      switch (stateKey) {
        case 'alertColor':
          // 正常Tag（默认色为#0077ff），另外定义变量区别于品牌色
          this.setVar('--o-color-state-primary', stateConfig[stateKey]);
          // 全局信息提示内容背景色,调和计算（mix mixColor 72%）
          color = new Color(stateConfig[stateKey]);
          contentColor = color.mixin(mixColor, 0.28);
          this.setVar('--o-color-state-primary-secondary', contentColor.value);
          break;
        case 'successColor':
          this.setVar('--o-color-success', stateConfig[stateKey]);

          // 全局信息成功提示内容背景色,调和计算（mix mixColor 72%）
          color = new Color(stateConfig[stateKey]);
          contentColor = color.mixin(mixColor, 0.28);
          this.setVar('--o-color-success-secondary', contentColor.value);
          break;
        default:
          break;
      }
    });
  }

  private setRadius(val: string) {
    switch (val) {
      case 'small':
        this.setVar('--o-border-radius', '4px');
        this.setVar('--o-border-radius-small', '2px');
        this.setVar('--i-button-border-radius', '2px');
        break;
      case 'middle':
        this.setVar('--o-border-radius', '6px');
        this.setVar('--o-border-radius-small', '4px');
        this.setVar('--i-button-border-radius', '4px');
        break;
      case 'large':
        this.setVar('--o-border-radius', '6px');
        this.setVar('--o-border-radius-small', '4px');
        this.setVar('--i-button-border-radius', '99px');
        break;
      default:
        break;
    }
  }

  private setShadow(val: string) {
    switch (val) {
      case 'small':
        this.setVar('--o-box-shadow', '0 4px 8px 0 rgba(0,0,0,0.2)');
        this.setVar('--el-box-shadow-dark', '-4px 4px 8px 0 rgba(0,0,0,0.2)');
        break;
      case 'middle':
        this.setVar('--o-box-shadow', '0 8px 16px 0 rgba(0,0,0,0.2)');
        this.setVar('--el-box-shadow-dark', '-16px 4px 16px 0 rgba(0,0,0,0.2)');
        break;
      case 'large':
        this.setVar('--o-box-shadow', '0 16px 32px 0 rgba(0,0,0,0.2)');
        this.setVar('--el-box-shadow-dark', '-24px 4px 24px 0 rgba(0,0,0,0.2)');
        break;
      default:
        break;
    }
  }
}
