//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
﻿
import $http from '@/utils/http-service';
import { traduction } from '@/utils/language';
import { isEmptyProperty } from '../../storage.service';

import { RaidNode, StorageTree, DriverNode, Driver } from '../../models';
import { unitConversion } from '../../utils';

export function factory(driverNode: DriverNode) {
  let raidNode: RaidNode;
  let getRaidPost$;
  const getDriverPost$ = getData(driverNode.url);
  return Promise.all([getRaidPost$, getDriverPost$]).then(
    ([raidPostData, driverPostData]) => {
      if (raidNode && raidPostData) {
        raidNode.setRaid = raidPostData;
      }
      return createDriver(driverPostData.data, driverNode.getIsRaid, driverNode);
    },
  );
}

export function createDriver(bodyData: any, isRaid: boolean, driverNode: DriverNode): Driver {
  const interType = isEmptyProperty(bodyData.Protocol);
  const healthState = bodyData.Health;
  const manufacturer = isEmptyProperty(bodyData.Manufacturer);
  const model = isEmptyProperty(bodyData.Model);
  const serialNumber = isEmptyProperty(bodyData.SerialNumber);
  const firmwareVersion = isEmptyProperty(bodyData.Revision);
  const mediaType = isEmptyProperty(bodyData.MediaType);
  const temperature = isEmptyProperty(bodyData.TemperatureCelsius, ' ℃');
  const firmwareStatus = isEmptyProperty(bodyData.FirmwareStatus);
  const sasAddress0 = isEmptyProperty(bodyData?.SASAddress ? bodyData?.SASAddress[0] : '');
  const sasAddress1 = isEmptyProperty(bodyData?.SASAddress ? bodyData?.SASAddress[1] : '');
  const capacityBytes = (bodyData.CapacityBytes === 0 || bodyData.CapacityBytes) ?
    unitConversion(bodyData.CapacityBytes, 4, 2) : null;
  const capacity = isEmptyProperty(capacityBytes, ' TB');
  const capableSpeed = isEmptyProperty(bodyData.CapableSpeedGbs, ' Gbps');
  const negotiatedSpeed = isEmptyProperty(bodyData.NegotiatedSpeedGbs, ' Gbps');
  const powerState = isEmptyProperty(bodyData.PowerState);
  const hotspareType = isEmptyProperty(bodyData.HotspareType);
  const estimatedRemainingLifespan = bodyData.EstimatedRemainingLifespan === 4294967294 ?
    '--' : isEmptyProperty(bodyData.EstimatedRemainingLifespan);
  const rebuildState = bodyData.RebuildState;
  let tmpRebuildState = null;
  if (rebuildState === 'DoneOrNotRebuilt') {
    tmpRebuildState = traduction('FDM_STOPPED');
  } else if (rebuildState === 'Rebuilding') {
    const rebuildProgress = bodyData.RebuildProgress === null ? '' : bodyData.RebuildProgress;
    tmpRebuildState = `${traduction('FDM_RECONSTRUCTION')}  ${rebuildProgress}`;
  }
  const construction = isEmptyProperty(tmpRebuildState);
  const patrolState = isEmptyProperty(bodyData.PatrolState);
  const indicator = isEmptyProperty(bodyData.IndicatorLED);
  const powerSupply = isEmptyProperty(bodyData.HoursOfPoweredUp, ' h');
  const wearRate = isEmptyProperty(bodyData.PredictedMediaLifeLeftPercent, '%');
  const partNum = isEmptyProperty(bodyData.PartNum);
  const resourceOwner = isEmptyProperty(bodyData.AssociatedResource);
  const diskSpeed = isEmptyProperty(bodyData.RotationSpeedRPM);
  const isEPD = Boolean(bodyData.IsEPD);
  const bootEnabled = bodyData.BootEnabled;  
  const bootPriority = bodyData.BootPriority || null;
  return new Driver(
    interType, healthState, manufacturer, model, serialNumber, firmwareVersion, mediaType, temperature,
    firmwareStatus, sasAddress0, sasAddress1, capacity, capableSpeed, negotiatedSpeed, powerState,
    hotspareType, construction, patrolState, indicator, powerSupply, wearRate, partNum, resourceOwner,
    isRaid, diskSpeed, isEPD, driverNode.getRaidType, bootEnabled, bootPriority, estimatedRemainingLifespan,
  );
}

export function getData(url: string) {
  return $http.get(url);
}

