//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
import { LANG } from '@/model/base-enum';
import { IParam, IZreo } from '../models/ntp.datatype';
import areaData from '@/assets/baseI18n/time-area-data/area-data';
import timeData from '@/assets/baseI18n/time-area-data/time-data';
/**
 * 获取接口返回当前的地区和时区
 */
export function getInitTime(ntpData: any) {
  let time: [string, string] = ['', ''];
  let check = ntpData.dateTimeLocalOffset;
  const ibmcTime = `(UTC${ntpData.timeZone})`;
  if (!check) {
    return { time };
  } else if (/^UTC/.test(check) || /^GMT/.test(check)) {
    time = [check, 'UTC'];
  } else {
    const index = check.indexOf('/');
    time = [check.substring(index + 1), check.slice(0, index)];
  }
  // 页脚时间偏移量刷新
  return { time, ibmcTime };
}

/**
 * 获取地区下拉列表
 * @param locale 语言
 */
export function getAreaList(locale: string) {
  return transform(areaData[LANG[locale]]).sort((param1, param2) => {
    return param1.label.localeCompare(param2.label);
  });
}

/**
 * 将json数据转化为需要的结构体
 * @param obj 接口返回值
 */
function transform(obj: Record<string, string>): IZreo[] {
  const arr: IZreo[] = [];
  for (const key in obj) {
    if (String(obj[key]) === String(0) || obj[key]) {
      arr.push({
        key,
        label: obj[key],
      });
    }
  }
  return arr;
}

/**
 * 获取当前时区下拉列表和下拉列表选中项
 * @param timeList 时区列表
 * @param area 地区返回值
 * @param time 时区返回值
 * @param isFromChangeArea
 */
export function getTimeSelect(timeList: IZreo[], area: string, time: string, isFromChangeArea: boolean = false): {
  time: IZreo;
  paramList: IParam[];
} {
  const paramList: IParam[] = timeData[area].areaList;
  let select: IParam = null as any;
  if (time.indexOf('UTC') > -1) {
    select = paramList.filter((obj: any) => obj.value === time || obj.content === time)[0];
  } else {
    select = paramList.filter(
      (obj: any) => obj.value.indexOf(time) > -1 || obj.content.indexOf(time) > -1,
    )[0];
  }
  let izero = select ? timeList.filter(val => val.key === select.content) : [];
  if (izero.length > 0) {
    return { time: izero[0], paramList };
  } else if (!isFromChangeArea && time.indexOf('GMT') > -1) {
    const gmtOption: IZreo = {
      key: time,
      label: time,
    };
    timeList.push(gmtOption);
    paramList.push({
      content: time,
      value: time,
    });
    return { time: gmtOption, paramList };
  } else {
    izero = select ? timeList.filter(val => val.key === select.relatedContent) : [];
    return { time: izero[0] || timeList[0], paramList };
  }
}

/**
 * 获取时区下拉列表
 * @param area 地区返回值
 * @param locale 语言
 */
export function getTimeList(area: string, locale: string) {
  const timeList = timeData[area][LANG[locale]];
  const arr =
    area === 'UTC'
      ? transform(timeList)
      : transform(timeList).sort((param1, param2) => {
        return param1.label.localeCompare(param2.label);
      });
  return arr;
}

export function changeArea(area: string, locale: string, timeOrigin: string) {
  const timeList = getTimeList(area, locale);
  const timeSelectData = getTimeSelect(timeList, area, timeOrigin, true);
  return { timeSelect: timeSelectData.time, timeList, paramList: timeSelectData.paramList };
}
