//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
/**
 * iBMC权限对应关系
 * QueryEnabled                查询功能权限
 * BasicConfigEnabled          常规配置权限
 * ConfigureSelfEnabled        配置自身权限
 * UserConfigEnabled           用户配置权限
 * RemoteControlEnabled        远程控制权限
 * SecurityConfigEnabled       安全配置权限
 * PowerControlEnabled         电源控制权限
 * VMMEnabled                  远程媒体权限
 * DiagnosisEnabled            调试诊断权限
 */

import { VideoId } from './../../../model/base-enum';
import { getStoreData } from '@/utils/composition';

const routeList = {
  '/login': {
    access: ['QueryEnabled'],
  },
  '/kvm_h5': {
    access: ['RemoteControlEnabled'],
  },
  '/navigate/home': {
    access: ['QueryEnabled'],
  },

  // 系统管理
  '/navigate/system/info': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/info/units': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/info/others': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/info/product': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/info/processor': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/info/memory': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/info/sensor': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/info/discrete-sensors': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/info/fru-info': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/info/net': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/monitor': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/storage': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/power': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/power/info': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/power/converter': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/power/capping': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/power/control': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/power/multi-control': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/power/energy': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/power/energy-saving': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/fans': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/fans/fans-info': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/fans/fans-sensor': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/fusion-par': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/bios-settings': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/bios-settings/bios': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/bios-settings/cpu': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/bios-settings/setting': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/battery': {
    access: ['QueryEnabled'],
  },
  '/navigate/system/asset': {
    access: ['QueryEnabled'],
  },

  // 维护诊断
  '/navigate/maintenance/alarm': {
    access: ['QueryEnabled'],
  },
  '/navigate/maintenance/report/syslog': {
    access: ['QueryEnabled'],
  },
  '/navigate/maintenance/report/syslogdb': {
    access: ['QueryEnabled'],
  },
  '/navigate/maintenance/report/email': {
    access: ['QueryEnabled'],
  },
  '/navigate/maintenance/report/trap': {
    access: ['QueryEnabled'],
  },
  '/navigate/maintenance/fdm': {
    access: ['QueryEnabled'],
  },
  '/navigate/maintenance/video': {
    access: ['DiagnosisEnabled'],
  },
  '/navigate/maintenance/diagnostics': {
    access: ['DiagnosisEnabled'],
  },
  '/navigate/maintenance/ibmc-logs': {
    access: ['QueryEnabled'],
  },
  '/navigate/maintenance/ibmc-logs/operation': {
    access: ['QueryEnabled'],
  },
  '/navigate/maintenance/ibmc-logs/run': {
    access: ['QueryEnabled'],
  },
  '/navigate/maintenance/ibmc-logs/security': {
    access: ['QueryEnabled'],
  },
  '/navigate/maintenance/work-records': {
    access: ['QueryEnabled'],
  },
  '/navigate/maintenance/asset-maintance': {
    access: ['QueryEnabled'],
  },
  '/navigate/maintenance/asset-maintance/asset-retire': {
    access: ['QueryEnabled'],
  },
  '/navigate/maintenance/asset-maintance/asset-change': {
    access: ['QueryEnabled'],
  },
  '/navigate/maintenance/asset-maintance/asset-inventory': {
    access: ['QueryEnabled'],
  },

  // 用户与安全
  '/navigate/user/local-users': {
    access: ['QueryEnabled'],
  },
  '/navigate/user/ldap': {
    access: ['QueryEnabled'],
  },
  '/navigate/user/kerberos': {
    access: ['QueryEnabled'],
  },
  '/navigate/user/two-factor/root': {
    access: ['UserConfigEnabled'],
  },
  '/navigate/user/two-factor/client': {
    access: ['UserConfigEnabled'],
  },
  '/navigate/user/online-users': {
    access: ['UserConfigEnabled'],
  },
  '/navigate/user/security': {
    access: ['QueryEnabled'],
  },
  '/navigate/user/tpcm': {
    access: ['QueryEnabled'],
  },
  '/navigate/user/security/enhance': {
    access: ['QueryEnabled'],
  },
  '/navigate/user/security/rule': {
    access: ['QueryEnabled'],
  },
  '/navigate/user/security/permission': {
    access: ['UserConfigEnabled'],
  },
  '/navigate/user/security/notice': {
    access: ['QueryEnabled'],
  },
  '/navigate/user/certificate': {
    access: ['QueryEnabled'],
  },
  '/navigate/user/certificate/auto-certificate-update': {
    access: ['QueryEnabled'],
  },
  '/navigate/user/certificate/ca-certificate-list': {
    access: ['QueryEnabled'],
  },
  '/navigate/user/certificate/ssl-certificate': {
    access: ['QueryEnabled'],
  },

  // 服务管理
  '/navigate/service/port': {
    access: ['QueryEnabled'],
  },
  '/navigate/service/web': {
    access: ['QueryEnabled'],
  },
  '/navigate/service/kvm': {
    access: ['QueryEnabled'],
  },
  '/navigate/service/vmm': {
    access: ['QueryEnabled'],
  },
  '/navigate/service/vnc': {
    access: ['QueryEnabled'],
  },
  '/navigate/service/snmp': {
    access: ['QueryEnabled'],
  },

  // 网络配置
  '/navigate/manager/network': {
    access: ['QueryEnabled'],
  },
  '/navigate/manager/ntp': {
    access: ['QueryEnabled'],
  },
  '/navigate/manager/upgrade': {
    access: ['BasicConfigEnabled'],
  },
  '/navigate/manager/config-update': {
    access: ['UserConfigEnabled'],
  },
  '/navigate/manager/language': {
    access: ['BasicConfigEnabled'],
  },
  '/navigate/manager/license': {
    access: ['QueryEnabled'],
  },
  '/navigate/manager/ibma': {
    access: ['QueryEnabled'],
  },
  '/navigate/manager/sp': {
    access: ['QueryEnabled'],
  },
  '/navigate/manager/sp/sp-manage': {
    access: ['QueryEnabled'],
  },
  '/navigate/manager/sp/sp-task': {
    access: ['QueryEnabled'],
  },
  '/navigate/manager/usb': {
    access: ['QueryEnabled'],
  },
};

// 由于Video播放器地址的url携带playId，所以拼接push到routeList
for (let key in VideoId) {
  if ({}.hasOwnProperty.call(VideoId, key)) {
    routeList[`/videoPlayer/${VideoId[key]}`] = { access: ['DiagnosisEnabled'] };
  }
}

// 检测路由路径是否有效
export function checkRouteIsValid(route: string = '') {
  return routeList[route] !== undefined;
}

// 检测用户是否有足够权限访问路由路由
export function checkHasPrivil(route: string) {
  let hasPrivil = true;
  const privils = getStoreData('loct', 'privil');
  const needPrivil = routeList[route].access;
  needPrivil.forEach((item: string) => {
    hasPrivil = hasPrivil && privils.includes(item);
  });
  return hasPrivil;
}

// 检测页面上某功能所需权限是否在用户权限列表内
export function checkPrivil(privils: string | string[]) {
  const userPrivil = getStoreData('loct', 'privil');
  // 支持字符串参数和字符串数组参数
  if (typeof privils === 'string') {
    return userPrivil.includes(privils);
  }

  if (Array.isArray(privils)) {
    let result = true;
    privils.forEach(item => {
      result = result && userPrivil.includes(item);
    });
    return result;
  }

  return false;
}
