-- Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
--
-- this file licensed under the Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
--
-- THIS SOFTWARE IS PROVIDED ON AN \"AS IS\" BASIS, WITHOUT WARRANTIES OF ANY KIND, EITHER EXPRESS OR
-- IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT, MERCHANTABILITY OR FIT FOR A PARTICULAR
-- PURPOSE.
-- See the Mulan PSL v2 for more details.

local class = require 'mc.class'
local log = require 'mc.logging'
local client = require 'thermal_mgmt.client'
local props = require 'basic_cooling.define.cooling_properties'
local mdb = require 'mc.mdb'
local base = require 'basic_cooling.data_keeping.base'
local pumps = require 'basic_cooling.cooling_device.cooling_pumps'
local utils = require 'basic_cooling.cooling_utils'

local pump_group_data_keeping = class(base)

function pump_group_data_keeping:init()
    self.obj_name = 'PumpGroup'
    -- 监听的资源树路径
    self.path = "/bmc/kepler/Chassis/:ChassisId/ThermalSubsystem/PumpGroups/:Id"
    -- 监听的接口
    self.interfaces_table = {
        ['bmc.kepler.Chassis.PumpGroup'] = client.ForeachPumpGroupObjects
    }
    -- 监听的属性
    self.props_table = {
        [props.EXPECTED_SPEED_PERCENT] = true
    }
    self.super.init(self)
    self.pumps_instance = pumps.get_instance()
    self.pump_control_table = {}
end

function pump_group_data_keeping:properties_changed()
    return client:OnPumpGroupPropertiesChanged(function (prop_value, path, interface)
        self:property_changed_callback(path, interface, prop_value)
    end)
end

function pump_group_data_keeping:action_after_property_changed(path, interface, prop, value)
    -- 获取修改的资源树对象
    local pump_group = mdb.get_object(self.bus, path, interface)

    -- 更新泵组转速
    self:update_pump_group_control_table(pump_group)
end

function pump_group_data_keeping:action_after_obj_added(path, interface)
    local pump_group = client:GetPumpGroupObjects()[path]

    self:update_pump_group_control_table(pump_group)
end

function pump_group_data_keeping:update_pump_group_control_table(pump_group)
    local pump_slots_table = {string.byte(pump_group.PumpSlots, 1, -1)}
    for _, pump_id in pairs(pump_slots_table) do
        self.pump_control_table[pump_id] = pump_group.ExpectedSpeedPercent
    end
    local policy_info = utils.parse_table_info(self.pump_control_table)
    log:notice("Update pump group(%s), control table speed percent(%s)", pump_group.Id, policy_info)
end

function pump_group_data_keeping:get_pump_group_control_table()
    return self.pump_control_table
end

return pump_group_data_keeping