-- Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
--
-- this file licensed under the Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
--
-- THIS SOFTWARE IS PROVIDED ON AN \"AS IS\" BASIS, WITHOUT WARRANTIES OF ANY KIND, EITHER EXPRESS OR
-- IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT, MERCHANTABILITY OR FIT FOR A PARTICULAR
-- PURPOSE.
-- See the Mulan PSL v2 for more details.

local skynet = require 'skynet'
local class = require 'mc.class'
local log = require 'mc.logging'

local cooling_ipmi_init = require 'basic_cooling.cooling_ipmi_init'
local plugin_method_init = require 'basic_cooling.plugin_method_init'
local cooling_mgmt = require 'basic_cooling.cooling_mgmt'
local cooling_objs = require 'basic_cooling.cooling_intf.cooling_objs'
local cooling_ipmi = require 'basic_cooling.cooling_intf.cooling_ipmi'
local cooling_utils = require 'basic_cooling.cooling_utils'
local exp_policy = require 'basic_cooling.exception_policy'
local fans_config = require 'basic_cooling.cooling_config.fans_config'
local pumps_config = require 'basic_cooling.cooling_config.pumps_config'
local cooling_rpc = require 'basic_cooling.cooling_intf.cooling_rpc'
local cooling_init_service = require 'basic_cooling.cooling_init_service'

local cooling_service = class(cooling_objs)

function cooling_service:ctor(service, db, bus, persist, register_ipmi_func)
    self.service = service
    self.persist = persist
    self.db = db
    self.bus = bus
    self.register_ipmi_func = register_ipmi_func
    -- 单例对象，先销毁再创建
    cooling_utils.destroy()
    exp_policy.destroy()
    fans_config.destroy()
    pumps_config.destroy()
end

function cooling_service:init()
    log:notice('********************** basic cooling initializing ********************************')
    self.super.init(self)
    self.cooling_utils = cooling_utils.new()
    self.cooling_init_service = cooling_init_service.new(self.cooling_flag)
    skynet.fork_once(function()
        while true do
            skynet.sleep(200)
            -- cooling组件启动的最基本的对象已经准备好且散热初始化服务执行完成
            if self.cooling_flag.is_base_obj_added and self.cooling_flag.cooling_init_service_is_end then
                skynet.sleep(200) -- 增加定制转速的观测时间
                cooling_init_service.destroy()
                self:start()
                self.is_initialized = true
                break
            end
        end
    end)
    log:notice('********************** basic cooling initialized ********************************')
end

function cooling_service:start()
    log:notice('********************** basic cooling starting ********************************')
    plugin_method_init.privite_method_register(self.service)
    self.exp_policy = exp_policy.new(self.service, self.cooling_data,
        self.extern_data.data_keeping_objs.fructl_data_keeping)

    self.cooling_mgmt = cooling_mgmt.new(self.service, self.db, self.bus, self.cooling_data)
    self.cooling_ipmi = cooling_ipmi.new(self.db, self.cooling_data)
    self.cooling_rpc = cooling_rpc.new(self.service, self.db, self.cooling_data)

    cooling_ipmi_init.register_ipmi(self.cooling_ipmi, self.register_ipmi_func)
    log:notice('********************** basic cooling started ********************************')
end

-- 平滑重启准备接口
function cooling_service:on_prepare()
    if self.cooling_mgmt then
        return self.cooling_mgmt:on_reboot_prepare()
    end
    return 0
end

-- 平滑重启接口
function cooling_service:on_action()
    if self.cooling_mgmt then
        return self.cooling_mgmt:on_reboot_action()
    end
    return 0
end

-- 平滑重启取消接口
function cooling_service:on_cancel()
    if self.cooling_mgmt then
        self.cooling_mgmt:on_reboot_cancel()
    end
end

return cooling_service
