-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local context = require 'mc.context'
local log = require 'mc.logging'
local cjson = require 'cjson'
local custom_messages = require 'messages.custom'

local SMS_INTF = 'bmc.kepler.Systems.Sms'
local SMS_PATH = '/bmc/kepler/Systems/1/Sms'
local m = {}
local HTTP_OK = 200

-- bma中返回的是string类型的数据，处理错误时需要额外处理，错误场景返回值如下
--[[
    err : {
        Response : { -- 返回的是string类型，需要解析为table
            ResponseStatusCode : 400,
            ResponseHeaders: {},
            ResponseBody : { -- 返回的是string类型，需要解析为table
                "status": 400,
                "message": ""
            }
        }
    }
]]
local function ibma_error(ctx, err, set_name)
    local message
    if err == nil or err.Response == nil then
        message = err == nil and 'err == nil' or 'err.Response == nil'
        log:error('Set %s config failed. err: %s', string.lower(set_name), message)
        log:operation(ctx:get_initiator(), 'web_backend', string.format('Set BMA %s config failed', set_name))
        error(custom_messages.OperationFailed())
    end

    local rsp = cjson.decode(err.Response)
    if rsp.ResponseStatusCode ~= HTTP_OK then
        message = rsp.ResponseBody and cjson.decode(rsp.ResponseBody).message or
            'not rsp.ResponseBody or not cjson.decode(rsp.ResponseBody).message'
        log:error('Set %s config failed. ResponseStatusCode:%s, err: %s',
            string.lower(set_name), rsp.ResponseStatusCode, message)
        log:operation(ctx:get_initiator(), 'web_backend', string.format('Set BMA %s config failed', set_name))
        error(custom_messages.OperationFailed())
    end
end

function m.set_base_config(input)
    local ctx = context.get_context()
    -- 请求体为空时不做处理
    if cjson.decode(input).RequestBody == '{}' then
        log:error("Set base config failed. err: body == '{}'")
        log:operation(ctx:get_initiator(), 'web_backend', 'Set BMA Base config failed')
        return
    end

    local sms_obj = mdb.get_object(bus, SMS_PATH, SMS_INTF)
    local ok, err = pcall(function()
        return sms_obj:ForwardRequest(ctx, input, 1)
    end)
    if not ok then
        log:error('Set base config failed. err: %s', err)
        log:operation(ctx:get_initiator(), 'web_backend', 'Set BMA Base config failed')
        return
    end
    ibma_error(ctx, err, 'Base')

    log:operation(ctx:get_initiator(), 'web_backend', 'Set BMA Base config successfully')
end

function m.set_logs_config(input)
    local ctx = context.get_context()
    -- 请求体为空时不做处理
    if cjson.decode(input).RequestBody == '{}' then
        log:error("Set logs config failed. err: body == '{}'")
        log:operation(ctx:get_initiator(), 'web_backend', 'Set Logs config failed')
        return
    end
    local sms_obj = mdb.get_object(bus, SMS_PATH, SMS_INTF)
    local ok, err = pcall(function()
        return sms_obj:ForwardRequest(ctx, input, 1)
    end)
    if not ok then
        log:error('Set logs config failed. err: %s', err)
        log:operation(ctx:get_initiator(), 'web_backend', 'Set BMA Logs config failed')
        return
    end
    ibma_error(ctx, err, 'Logs')

    log:operation(ctx:get_initiator(), 'web_backend', 'Set BMA Logs config successfully')
end

function m.set_monitor_config(input)
    local ctx = context.get_context()
    if input == nil then
        log:error("Set monitor failed. err: input == nil")
        log:operation(ctx:get_initiator(), 'web_backend', 'Set BMA Monitor config failed')
        error(custom_messages.OperationFailed())
    end
    local sms_obj = mdb.get_object(bus, SMS_PATH, SMS_INTF)
    local ok, err = pcall(function()
        return sms_obj:ForwardRequest(ctx, input, 1)
    end)
    if not ok then
        log:error('Set monitor config failed. err: %s', err)
        log:operation(ctx:get_initiator(), 'web_backend', 'Set BMA Monitor config failed')
        return
    end

    ibma_error(ctx, err, 'Monitor')
    log:operation(ctx:get_initiator(), 'web_backend', 'Set BMA Monitor config successfully')
end

return m
