-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local cjson = require 'cjson'
local log = require 'mc.logging'
local detection_path = '/bmc/kepler/Chassis/1/ThermalSubsystem/LeakDetection'
local detection_interface = 'bmc.kepler.Chassis.LeakDetection'

local FruStateEnums = {
    ["OFF"]="Off",
    ["ONING"]="Off",
    ["ON"]="On",
    ["OFFING"]="On"
}

local m = {}

local function ismultihost()
    local ok, obj = pcall(mdb.get_object, bus, "/bmc/kepler/Managers/1/Multihost", "bmc.kepler.Managers.Multihost")
    if not ok then
        log:error("MultiHost object is not configured")
        return false
    end
    return obj["HostType"] == "Multihost"
end

-- 获取是否支持风扇特性
function m.get_fan_supported(input)
    -- 获取不到路径下资源树属性时候返回为nil
    if not input then
        return false
    end
    local device_specication = cjson.json_object_to_table(input)
    if type(device_specication) ~= 'table' then
        return false
    end
    for _, device in pairs(device_specication) do
        -- 下标1为设备名字，下标2为支持的设备最大个数
        if device[1] == 'Fan' then
            if device[2] > 0 then
                return true
            else
                return false
            end
        end
    end
    return false
end

-- 是否支持漏液检测
function m.get_leakdetect_support()
    -- 确认资源树对象
    local ok, obj = pcall(mdb.get_object, bus, detection_path, detection_interface)
    local prop_name = 'LeakStrategy'
    local err = custom_messages.PropertyModificationNotSupported(prop_name)
    err.RelatedProperties = {'#/' .. prop_name}

    if not ok then
        log:error('Get leak detection obj failed, err_msg: %s', obj)
        error(err)
    end

    -- 确认是否支持漏液下电设置动作
    if obj.Enabled ~= true then
        log:error("Don't support leakage detection")
        error(err)
    end
    return true
end

function m.getpowerstate()
    if ismultihost() then
        local ok, obj = pcall(mdb.get_object, bus, "/bmc/kepler/Chassis/1/FruCtrl", "bmc.kepler.Chassis.FruCtrl")
        if not ok then
            log:error("ChassisFruCtrl is not configured")
            return
        end
        return FruStateEnums[obj.PowerState]
    end
    local ok, frulist = pcall(mdb.get_sub_objects, bus, "/bmc/kepler/Systems/1/FruCtrl", "bmc.kepler.Systems.FruCtrl")
    if not ok then
        log:error("FruCtrl is not configured")
        return
    end
    for _, fruobj in pairs(frulist) do
        return FruStateEnums[fruobj.PowerState]
    end
end

return m
