-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.
local utils = require 'mc.utils'
local m = {}

local valid_log_type = {
    'OperationLog', 'SecurityLog', 'EventLog', 'IntrusionDetectionLog'
}

local function check_log_type_valid(log_type)
    for _, v in ipairs(log_type) do
        if not utils.array_contains(valid_log_type, v) then
            return false
        end
    end
    return true
end

local function split_log_type(log_type, comma_cnt)
    local log_type_tab = {}
    if comma_cnt == 0 then
        table.insert(log_type_tab, log_type:sub(1, -2))
    elseif comma_cnt == 1 then
        local log1, log2 = log_type:match('(%a+),(%a+)')
        if log1 ~= nil and log2 ~= nil then
            table.insert(log_type_tab, log1:sub(1, -2))
            table.insert(log_type_tab, log2:sub(1, -2))
        else
            error(base_messages.InternalError())
        end
    elseif comma_cnt == 2 then
        local log1, log2, log3 = log_type:match('(%a+),(%a+),(%a+)')
        if log1 ~= nil and log2 ~= nil and log3 ~= nil then
            table.insert(log_type_tab, log1:sub(1, -2))
            table.insert(log_type_tab, log2:sub(1, -2))
            table.insert(log_type_tab, log3:sub(1, -2))
        else
            error(base_messages.InternalError())
        end
    elseif comma_cnt == 3 then
        local log1, log2, log3, log4 = log_type:match('(%a+),(%a+),(%a+),(%a+)')
        if log1 ~= nil and log2 ~= nil and log3 ~= nil and log4 ~= nil then
            table.insert(log_type_tab, log1:sub(1, -2))
            table.insert(log_type_tab, log2:sub(1, -2))
            table.insert(log_type_tab, log3:sub(1, -2))
            table.insert(log_type_tab, log4:sub(1, -2))
        else
            error(base_messages.InternalError())
        end
    else
        error(base_messages.InternalError())
    end
    return log_type_tab
end

function m.get_log_type_tab(log_type)
    if log_type == 'all' then
        return {'OperationLog', 'SecurityLog', 'EventLog', 'IntrusionDetectionLog'}
    end

    if log_type == 'none' or log_type == '' then
        return {}
    end

    local comma_cnt = 0
    for i = 1, #log_type do
        if log_type:sub(i, i) == ',' then
            comma_cnt = comma_cnt + 1
        end
    end

    local log_type_tab = {}
    log_type_tab = split_log_type(log_type, comma_cnt)

    if not check_log_type_valid(log_type_tab) then
        error(base_messages.InternalError())
    end

    return log_type_tab
end

function m.get_log_type_str(log_type)
    if #log_type == 0 then
        return 'none'
    end

    if #log_type == 4 and check_log_type_valid(log_type) then
        return 'all'
    end

    local log_type_tab = {
        ['OperationLog'] = 0, ['SecurityLog'] = 0, ['EventLog'] = 0, ['IntrusionDetectionLog'] = 0
    }

    for _, v in pairs(log_type) do
        if utils.array_contains(valid_log_type, v) then
            log_type_tab[v] = log_type_tab[v] + 1
        end
    end

    local res = ''
    for k, v in pairs(log_type_tab) do
        if v == 1 then
            res = res .. k .. 's,'
        end
    end

    return res:sub(1, -2)
end

function m.parse_cert_info(input)
    local delim = ';'
    local pos = string.find(input, delim) -- 查找第一个分号的位置
    if not pos then
        error(base_messages.PropertyValueFormatError('Content', '******'))
    else
        local name = string.sub(input, 1, pos - 1)
        local password = string.sub(input, pos + 1)
        if #password > 127 then
            error(base_messages.PropertyValueFormatError('Content', '******'))
        end
        return {name, password}
    end
end

function m.get_max_ca_cert_id(ca_cert_info)
    local max_cert_id = 0
    for _, cert_path in ipairs(ca_cert_info) do
        local cert_id = tonumber(string.match(cert_path, "%d+$"))
        max_cert_id = max_cert_id > cert_id and max_cert_id or cert_id
    end

    return max_cert_id
end

return m
