-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.
local utils = require 'mc.utils'


-- 综合V2的snmp固件类型和snmp文档
local SNMP_FIRMWARE_ACTIVE_BMC <const> = 1
local SNMP_FIRMWARE_BACKUP_BMC <const> = 2
local SNMP_FIRMWARE_CPLD <const> = 3
local SNMP_FIRMWARE_BIOS <const> = 4
local SNMP_FIRMWARE_UBOOT <const> = 5
local SNMP_FIRMWARE_LCD <const> = 6
local SNMP_FIRMWARE_BAK_UBOOT <const> = 7
local SNMP_FIRMWARE_IOBOARD_CPLD <const> = 8
local SNMP_FIRMWARE_CPUBOARD_CPLD <const> = 9
local SNMP_FIRMWARE_HDD_BACKPLANE_CPLD <const> = 10
local SNMP_FIRMWARE_PCIE_RISERCARD_CPLD <const> = 11
local SNMP_FIRMWARE_RAID_CARD_CPLD <const> = 12
local SNMP_FIRMWARE_PERIPHERAL <const> = 13
local SNMP_FIRMWARE_GPUBOARD_CPLD <const> = 14
local SNMP_FIRMWARE_AVAILABLE_BMC <const> = 15
local SNMP_FIRMWARE_NPUBOARD_CPLD <const> = 16
local SNMP_FIRMWARE_CDR_5902L <const> = 17
local SNMP_FIRMWARE_RETIMER_5902H <const> = 18
local SNMP_FIRMWARE_EXPBOARD_CPLD <const> = 19
local SNMP_FIRMWARE_FANBOARD_CPLD <const> = 20
local SNMP_FIRMWARE_PEUBOARD_CPLD <const> = 21

local firmware_type2snmp_type = {
    ActiveBMC = SNMP_FIRMWARE_ACTIVE_BMC,
    ActiveUboot = SNMP_FIRMWARE_UBOOT,
    AvailableBMC = SNMP_FIRMWARE_AVAILABLE_BMC,
    BackupBMC = SNMP_FIRMWARE_BACKUP_BMC,
    BackupUboot = SNMP_FIRMWARE_BAK_UBOOT,
    Bios = SNMP_FIRMWARE_BIOS,
    BCU_CPLD = SNMP_FIRMWARE_CPUBOARD_CPLD,
    CLU_CPLD = SNMP_FIRMWARE_FANBOARD_CPLD,
    EXU_CPLD = SNMP_FIRMWARE_EXPBOARD_CPLD,
    SEU_CPLD = SNMP_FIRMWARE_HDD_BACKPLANE_CPLD,
    PEU_CPLD = SNMP_FIRMWARE_PEUBOARD_CPLD
}

local DEFAULT_FRU_ID <const> = 0

local m = {}

function m.is_snmp_firmware(id)
    for fw_type, _ in pairs(firmware_type2snmp_type) do
        if string.match(id, '(' .. fw_type .. ')') then
            return true
        end
    end
    if string.match(id, '(CPLD)') then
        return true
    end

    return false
end

function m.get_firmware_type(id)
    for key, value in pairs(firmware_type2snmp_type) do
        if string.match(id, '(' .. key .. ')') then
            return value
        end
    end
    if string.match(id, '(CPLD)') then
        -- 细分的CPLD类型未匹配，则归入CPLD大类
        return SNMP_FIRMWARE_CPLD
    end

    return SNMP_FIRMWARE_ACTIVE_BMC
end

local function get_board_prop(id, board_paths, prop)
    local pos
    if string.match(id, 'Bios') then
        local bios_n = string.match(id, 'Bios(%d)') or '1'  -- 兼容multihost场景
        pos = mdb.get_object(bus, '/bmc/kepler/Systems/' .. bios_n .. '/Bios',
            'bmc.kepler.Object.Properties').ObjectIdentifier[4]
    else
        local id_tab = utils.string_split(id, '_', true)
        pos = id_tab[#id_tab]
    end
    for _, v in ipairs(board_paths) do
        if pos == mdb.get_object(bus, v, 'bmc.kepler.Object.Properties').ObjectIdentifier[4] then
            return mdb.get_object(bus, v, 'bmc.kepler.Systems.Board')[prop]
        end
    end
end

function m.get_fru_number(id, board_paths)
    return get_board_prop(id, board_paths, 'FruID') or DEFAULT_FRU_ID
end

function m.get_firmware_board(id, board_paths)
    return get_board_prop(id, board_paths, 'Name') or 'BMC'
end

return m
