-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.


local class = require 'mc.class'
local log = require 'mc.logging'
local initiator = require 'mc.initiator'
local enums = require 'macros.power_mgmt_enums'
local singleton = require 'mc.singleton'
local utils = require 'mc.utils'
local parser_cfg = require 'parser_cfg'

local power_mgmt_utils = class(utils)

function power_mgmt_utils:ctor()
    self.last_time_map = {}
end

function power_mgmt_utils.operation_log(ctx, fmt, ...)
    local intf = (ctx and ctx.Interface) and ctx.Interface or 'N/A'
    local user_name = (ctx and ctx.UserName) and ctx.UserName or 'N/A'
    local client_addr = (ctx and ctx.ClientAddr)  and ctx.ClientAddr or '127.0.0.1'
    log:operation(initiator.new(intf, user_name, client_addr), 'power_mgmt', fmt, ...)
end

-- 限频日志
---@param level number params: 1,error 2,info 3,debug
---@param interval_s number params: interval seconds
function power_mgmt_utils:frequency_limit_log(level, interval_s, fmt, ...)
    local last_time_key = debug.getinfo(2, 'S').short_src .. ':' .. debug.getinfo(2, "l").currentline
    local cur_time = os.time()
    local log_callback = {
        [enums.LOG_LEVEL.ERROR] = log.error,
        [enums.LOG_LEVEL.INFO] = log.info,
        [enums.LOG_LEVEL.DEBUG] = log.debug,
        [enums.LOG_LEVEL.NOTICE] = log.notice
    }

    local last_time = self.last_time_map[last_time_key]
    if not last_time or cur_time - last_time >= interval_s then
        self.last_time_map[last_time_key] = cur_time
        log_callback[level](log, fmt, ...)
    end
end

local cfgs = {}

function cfgs:get_cfgs(file_path)
    local ok, version, cfg = pcall(parser_cfg.get_cfgs, file_path)
    if ok then
        self.version = version
        self.cfgs = cfg
    end
end

function cfgs:check_idex(idex)
    if not self.cfgs then
        return false
    end
    for _, cfg in pairs(self.cfgs) do
        if cfg then
            log:notice('power upgrade, component_idex %s', cfg.component_idex)
            if not cfg.component_idex then
                goto continue
            end
            if cfg.component_idex == idex then
                return true
            end
        end
        ::continue::
    end
    return false
end

function cfgs:get_component_idex()
    if not self.cfgs then
        log:notice('get_component_idex cfg nil')
        return 0
    end
    for _, cfg in pairs(self.cfgs) do
        if cfg then
            log:notice('power upgrade, component_idex %s', cfg.component_idex)
            if not cfg.component_idex then
                goto continue
            end
            if cfg.component_idex == 2 then  -- 2代表canbus电源
                log:notice('canbus power upgrade')
                return cfg.component_idex
            end
        end
        ::continue::
    end
    return 0  -- 默认按照pmbus电源处理
end

power_mgmt_utils.cfgs = cfgs

return singleton(power_mgmt_utils)