-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local class = require 'mc.class'
local utils = require 'device_tree.adapters.power_mgmt.utils'
local utils_core = require 'utils.core'
local ctx = require 'mc.context'
local log = require 'mc.logging'

local READ_FAIL_DEBOUNCE_COUNT <const> = 3
local E_OK = nil -- 函数执行成功返回nil
local E_FAILED = '' -- 空错误信息

local function get_pbit(val)
    return 1 << val
end

return function (obj)
    local psu_slot = class(obj)
    -- qb900
    function psu_slot:ctor(protocol)
        self.protocol = protocol
        self.read_fail_count = 0
        self.last_read_interval = 10
        self.read_interval = 300
        self.monitor_tasks = {}
    end

    function psu_slot:add_monitor_tasks(cb)
        table.insert(self.monitor_tasks, self:next_tick(cb))
    end

    function psu_slot:stop_monitor_tasks()
        local ts = self.monitor_tasks
        self.monitor_tasks = {}
        for _, task in pairs(ts) do
            task:stop()
        end
    end

    function psu_slot:psm_monitor_stop()
        self:stop_monitor_tasks()
    end

    function psu_slot:fetch_power_supply_info()
        self:get_power_supply_info('manufacturer')
        self:get_power_supply_info('model')
        self:get_power_supply_info('production_date')
        self:get_power_supply_info('product_version')
        self:get_power_supply_info('firmware_version')
        self:get_power_supply_info('serial_number')
        self:get_power_supply_info('part_number')
    end

    function psu_slot:update_health_event()
        local health_event, failed_flag = self.protocol:get_health_event()
        if not health_event then
            log:info('get health_event failed')
            return E_FAILED
        end

        if failed_flag then
            self:dal_refresh_property('health', get_pbit(4))
            if self.read_fail_count < READ_FAIL_DEBOUNCE_COUNT then
                self.read_fail_count = self.read_fail_count + 1
            else
                self:dal_refresh_property('communication_status', 1)
            end
        else
            self.read_fail_count = 0
            self:dal_refresh_property('communication_status', 0)
            self:dal_refresh_property('health', health_event.event)
        end

        self:dal_refresh_property('health_event', health_event)
        return E_OK
    end

    -- power_monitor 实际功率任务
    function psu_slot:power_monitor()
        if self:update_health_event() ~= E_OK then
            -- 接口调用失败， 获取不到属性，直接返回
            return E_FAILED
        end

        return E_OK
    end

    function psu_slot:psm_task_power()
        while true do
            self:power_monitor()
            self.scan_status = self.scan_status | get_pbit(0)
            self:dal_refresh_property('scan_status', self.scan_status)
            self:sleep_ms(15000)
        end
    end

    -- 查询电源输入/输出 电流/电压 等信息
    function psu_slot:psm_task_v_i()
        while true do
            self:get_dynamic_data('output_power_watts')
            self:sleep_ms(500)
            self:get_dynamic_data('input_current_amps')
            self:sleep_ms(500)
            self:get_dynamic_data('output_current_amps')
            self:sleep_ms(500)
            self:get_dynamic_data('input_voltage')
            self:sleep_ms(500)
            self:get_dynamic_data('output_voltage')
            self:sleep_ms(500)
            self:get_dynamic_data('power_supply_type')
            self:sleep_ms(500)
            self:get_dynamic_data('env_temperature_celsius')
            self:sleep_ms(500)
            self:get_dynamic_data('primary_chip_temperature_celsius')
            self:sleep_ms(500)
            self:get_dynamic_data('secondary_chip_temperature_celsius')

            self.scan_status = self.scan_status | get_pbit(2)
            self:dal_refresh_property('scan_status', self.scan_status)
            self:sleep_ms(15000)
        end
    end

    function psu_slot:power_monitor_entry()
        self:add_monitor_tasks(
            function()
                self:psm_task_power()
            end
        )
        self:add_monitor_tasks(
            function()
                self:psm_task_v_i()
            end
        )
        self.scan_status = self.scan_status | get_pbit(1) | get_pbit(3)
        self:dal_refresh_property('scan_status', self.scan_status)
    end

    function psu_slot:power_monitor_start()
        log:notice('ps%d power monitor start', self.ps_id)
        self:power_monitor_entry()
    end

    return psu_slot
end