-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

-- 光模块相关硬件配置
local IDENTIFIER_DEFAULT<const> = 0x00
local IDENTIFIER_TYPE_SFP<const> = 0x03
local IDENTIFIER_TYPE_QSFP28<const> = 0x11
local IDENTIFIER_TYPE_CMIS<const> = 0x1E
OpticalModuleConfig = {}
-- 数字结果保留2位小数
local function num_convert(input)
    return tonumber(string.format("%.2f",input))
end

OpticalModuleConfig.static_info_list = {
    [IDENTIFIER_TYPE_SFP] = {
        Manufacturer = {
            offset = 20, length = 16
        },
        PartNumber = {
            offset = 40, length = 16
        },
        SerialNumber = {
            offset = 68, length = 16
        },
        ProductionDate = {
            offset = 84, length = 8,
            handle_func = function(input)
                if input then
                    return string.format('20%s-%s-%s',
                        string.sub(input, 1, 2),
                        string.sub(input, 3, 4),
                        string.sub(input, 5, 6))
                end
            end
        },
        MediumType = {
            offset = 2, length = 1,
            handle_func = function(input)
                if input == 0x02 or input == 0x03 or input == 0x21 then
                    return "Copper"  -- 非光纤模块
                elseif input then
                    return "Optical"  -- 光纤模块
                end
            end
        }
    },
    [IDENTIFIER_TYPE_QSFP28] = {
        Manufacturer = {
            offset = 148, length = 16
        },
        PartNumber = {
            offset = 168, length = 16
        },
        SerialNumber = {
            offset = 196, length = 16
        },
        ProductionDate = {
            offset = 212, length = 8,
            handle_func = function(input)
                if input then
                    return string.format('20%s-%s-%s',
                        string.sub(input, 1, 2),
                        string.sub(input, 3, 4),
                        string.sub(input, 5, 6))
                end
            end
        },
        MediumType = {
            offset = 130, length = 1,
            handle_func = function(input)
                if input == 0x02 or input == 0x03 or input == 0x21 then
                    return "Copper"  -- 非光纤模块
                elseif input then
                    return "Optical"  -- 光纤模块
                end
            end
        }
    },
    [IDENTIFIER_TYPE_CMIS] = {
        Manufacturer = {
            offset = 129, length = 16
        },
        PartNumber = {
            offset = 148, length = 16
        },
        SerialNumber = {
            offset = 166, length = 16
        },
        ProductionDate = {
            offset = 182, length = 8,
            handle_func = function(input)
                if input then
                    return string.format('20%s-%s-%s',
                        string.sub(input, 1, 2),
                        string.sub(input, 3, 4),
                        string.sub(input, 5, 6))
                end
            end
        },
        MediumType = {
            offset = 85, length = 1,
            handle_func = function(input)
                if input == 0x03 then
                    return "Copper"  -- 非光纤模块
                elseif input then
                    return "Optical"  -- 光纤模块
                end
            end
        }
    }
}

OpticalModuleConfig.diagnostic_info_list = {
    [IDENTIFIER_TYPE_SFP] = {
        ReadingCelsius = {
            offset = 352, length = 2,
            handle_func = function(input)
                input = num_convert(input / 256)
                return (input < 128) and input or (input - 256)
            end
        },
        SupplyVoltage = {
            offset = 354, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        TXBiasCurrentMilliAmps = {
            Channel1 = {
                offset = 356, length = 2, alias = 'BiasCurrentMilliAmps',
                handle_func = function(input)
                    return num_convert(input / 500)
                end
            },
            UpperThreshold = 'BC_TXUpperThresholdCritical',
            LowerThreshold = 'BC_TXLowerThresholdCritical'
        },
        RXInputPowerMilliWatts = {
            Channel1 = {
                offset = 360, length = 2, alias = 'RXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            UpperThreshold = 'RXUpperThresholdCritical',
            LowerThreshold = 'RXLowerThresholdCritical'
        },
        TXOutputPowerMilliWatts = {
            Channel1 = {
                offset = 358, length = 2, alias = 'TXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            UpperThreshold = 'Power_TXUpperThresholdCritical',
            LowerThreshold = 'Power_TXLowerThresholdCritical'
        },
        RxLossState = {
            offset = 366, length = 1, mask = 2,
            handle_func = function(input)
                if input == 0 then
                    return false
                else
                    return true
                end
            end
        },
        TxLossState = {},
        TxFaultState = {
            offset = 366, length = 1, mask = 4,
            handle_func = function(input)
                if input == 0 then
                    return false
                else
                    return true
                end
            end
        },
        Temp_UpperThresholdCritical = {
            offset = 256, length = 2,
            handle_func = function(input)
                input = num_convert(input / 256)
                return (input < 128) and input or (input - 256)
            end
        },
        TemperatureUpperThresholdCritical = {
            offset = 256, length = 2,
            handle_func = function(input)
                input = num_convert(input / 256)
                return (input < 128) and input or (input - 256)
            end
        },
        Temp_LowerThresholdCritical = {
            offset = 258, length = 2,
            handle_func = function(input)
                input = num_convert(input / 256)
                return (input < 128) and input or (input - 256)
            end
        },
        TemperatureLowerThresholdCritical = {
            offset = 258, length = 2,
            handle_func = function(input)
                input = num_convert(input / 256)
                return (input < 128) and input or (input - 256)
            end
        },
        Vol_UpperThresholdCritical = {
            offset = 264, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        VoltageUpperThresholdCritical = {
            offset = 264, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        Vol_LowerThresholdCritical = {
            offset = 266, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        VoltageLowerThresholdCritical = {
            offset = 266, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        BC_TXUpperThresholdCritical = {
            offset = 272, length = 2,
            handle_func = function(input)
                return num_convert(input / 500)
            end
        },
        BC_TXLowerThresholdCritical = {
            offset = 274, length = 2,
            handle_func = function(input)
                return num_convert(input / 500)
            end
        },
        RXUpperThresholdCritical = {
            offset = 288, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        RXLowerThresholdCritical = {
            offset = 290, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        Power_TXUpperThresholdCritical = {
            offset = 280, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        Power_TXLowerThresholdCritical = {
            offset = 282, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        }
    },
    [IDENTIFIER_TYPE_QSFP28] = {
        ReadingCelsius = {
            offset = 22, length = 2,
            handle_func = function(input)
                input = num_convert(input / 256)
                return (input < 128) and input or (input - 256)
            end
        },
        SupplyVoltage = {
            offset = 26, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        TXBiasCurrentMilliAmps = {
            Channel1 = {
                offset = 42, length = 2, alias = 'BiasCurrentMilliAmps',
                handle_func = function(input)
                    return num_convert(input / 500)
                end
            },
            Channel2 = {
                offset = 44, length = 2, alias = 'BiasCurrentMilliAmps',
                handle_func = function(input)
                    return num_convert(input / 500)
                end
            },
            Channel3 = {
                offset = 46, length = 2, alias = 'BiasCurrentMilliAmps',
                handle_func = function(input)
                    return num_convert(input / 500)
                end
            },
            Channel4 = {
                offset = 48, length = 2, alias = 'BiasCurrentMilliAmps',
                handle_func = function(input)
                    return num_convert(input / 500)
                end
            },
            UpperThreshold = 'BC_TXUpperThresholdCritical',
            LowerThreshold = 'BC_TXLowerThresholdCritical'
        },
        RXInputPowerMilliWatts = {
            Channel1 = {
                offset = 34, length = 2, alias = 'RXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            Channel2 = {
                offset = 36, length = 2, alias = 'RXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            Channel3 = {
                offset = 38, length = 2, alias = 'RXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            Channel4 = {
                offset = 40, length = 2, alias = 'RXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            UpperThreshold = 'RXUpperThresholdCritical',
            LowerThreshold = 'RXLowerThresholdCritical'
        },
        TXOutputPowerMilliWatts = {
            Channel1 = {
                offset = 50, length = 2, alias = 'TXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            Channel2 = {
                offset = 52, length = 2, alias = 'TXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            Channel3 = {
                offset = 54, length = 2, alias = 'TXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            Channel4 = {
                offset = 56, length = 2, alias = 'TXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            UpperThreshold = 'Power_TXUpperThresholdCritical',
            LowerThreshold = 'Power_TXLowerThresholdCritical'
        },
        RxLossState = {
            offset = 3, length = 1, mask = 15,
            handle_func = function(input)
                if input == 0 then
                    return false
                else
                    return true
                end
            end
        },
        TxLossState = {
            offset = 3, length = 1, mask = 240,
            handle_func = function(input)
                if input == 0 then
                    return false
                else
                    return true
                end
            end
        },
        TxFaultState = {
            offset = 4, length = 1, mask = 15,
            handle_func = function(input)
                if input == 0 then
                    return false
                else
                    return true
                end
            end
        },
        Temp_UpperThresholdCritical = {
            offset = 512, length = 2,
            handle_func = function(input)
                input = num_convert(input / 256)
                return (input < 128) and input or (input - 256)
            end
        },
        TemperatureUpperThresholdCritical = {
            offset = 512, length = 2,
            handle_func = function(input)
                input = num_convert(input / 256)
                return (input < 128) and input or (input - 256)
            end
        },
        Temp_LowerThresholdCritical = {
            offset = 514, length = 2,
            handle_func = function(input)
                input = num_convert(input / 256)
                return (input < 128) and input or (input - 256)
            end
        },
        TemperatureLowerThresholdCritical = {
            offset = 514, length = 2,
            handle_func = function(input)
                input = num_convert(input / 256)
                return (input < 128) and input or (input - 256)
            end
        },
        Vol_UpperThresholdCritical = {
            offset = 528, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        VoltageUpperThresholdCritical = {
            offset = 528, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        Vol_LowerThresholdCritical = {
            offset = 530, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        VoltageLowerThresholdCritical = {
            offset = 530, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        BC_TXUpperThresholdCritical = {
            offset = 568, length = 2,
            handle_func = function(input)
                return num_convert(input / 500)
            end
        },
        BC_TXLowerThresholdCritical = {
            offset = 570, length = 2,
            handle_func = function(input)
                return num_convert(input / 500)
            end
        },
        RXUpperThresholdCritical = {
            offset = 560, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        RXLowerThresholdCritical = {
            offset = 562, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        Power_TXUpperThresholdCritical = {
            offset = 576, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        Power_TXLowerThresholdCritical = {
            offset = 578, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        }
    },
    [IDENTIFIER_TYPE_CMIS] = {
        ReadingCelsius = {
            offset = 14, length = 2,
            handle_func = function(input)
                input = num_convert(input / 256)
                return (input < 128) and input or (input - 256)
            end
        },
        SupplyVoltage = {
            offset = 16, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        TXBiasCurrentMilliAmps = {
            Channel1 = {
                offset = 2346, length = 2, alias = 'BiasCurrentMilliAmps',
                handle_func = function(input)
                    return num_convert(input / 500)
                end
            },
            Channel2 = {
                offset = 2348, length = 2, alias = 'BiasCurrentMilliAmps',
                handle_func = function(input)
                    return num_convert(input / 500)
                end
            },
            Channel3 = {
                offset = 2350, length = 2, alias = 'BiasCurrentMilliAmps',
                handle_func = function(input)
                    return num_convert(input / 500)
                end
            },
            Channel4 = {
                offset = 2352, length = 2, alias = 'BiasCurrentMilliAmps',
                handle_func = function(input)
                    return num_convert(input / 500)
                end
            },
            UpperThreshold = 'BC_TXUpperThresholdCritical',
            LowerThreshold = 'BC_TXLowerThresholdCritical'
        },
        RXInputPowerMilliWatts = {
            Channel1 = {
                offset = 2362, length = 2, alias = 'RXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            Channel2 = {
                offset = 2364, length = 2, alias = 'RXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            Channel3 = {
                offset = 2366, length = 2, alias = 'RXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            Channel4 = {
                offset = 2368, length = 2, alias = 'RXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            UpperThreshold = 'RXUpperThresholdCritical',
            LowerThreshold = 'RXLowerThresholdCritical'
        },
        TXOutputPowerMilliWatts = {
            Channel1 = {
                offset = 2330, length = 2, alias = 'TXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            Channel2 = {
                offset = 2332, length = 2, alias = 'TXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            Channel3 = {
                offset = 2334, length = 2, alias = 'TXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            Channel4 = {
                offset = 2336, length = 2, alias = 'TXPowerMilliWatts',
                handle_func = function(input)
                    return num_convert(input / 10000)
                end
            },
            UpperThreshold = 'Power_TXUpperThresholdCritical',
            LowerThreshold = 'Power_TXLowerThresholdCritical'
        },
        RxLossState = {
            offset = 2323, length = 1, mask = 15,
            handle_func = function(input)
                if input == 0 then
                    return false
                else
                    return true
                end
            end
        },
        TxLossState = {
            offset = 2312, length = 1, mask = 15,
            handle_func = function(input)
                if input == 0 then
                    return false
                else
                    return true
                end
            end
        },
        TxFaultState = {
            offset = 2311, length = 1, mask = 15,
            handle_func = function(input)
                if input == 0 then
                    return false
                else
                    return true
                end
            end
        },
        Temp_UpperThresholdCritical = {
            offset = 384, length = 2,
            handle_func = function(input)
                input = num_convert(input / 256)
                return (input < 128) and input or (input - 256)
            end
        },
        TemperatureUpperThresholdCritical = {
            offset = 384, length = 2,
            handle_func = function(input)
                input = num_convert(input / 256)
                return (input < 128) and input or (input - 256)
            end
        },
        Temp_LowerThresholdCritical = {
            offset = 386, length = 2,
            handle_func = function(input)
                input = num_convert(input / 256)
                return (input < 128) and input or (input - 256)
            end
        },
        TemperatureLowerThresholdCritical = {
            offset = 386, length = 2,
            handle_func = function(input)
                input = num_convert(input / 256)
                return (input < 128) and input or (input - 256)
            end
        },
        Vol_UpperThresholdCritical = {
            offset = 392, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        VoltageUpperThresholdCritical = {
            offset = 392, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        Vol_LowerThresholdCritical = {
            offset = 394, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        VoltageLowerThresholdCritical = {
            offset = 394, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        BC_TXUpperThresholdCritical = {
            offset = 440, length = 2,
            handle_func = function(input)
                return num_convert(input / 500)
            end
        },
        BC_TXLowerThresholdCritical = {
            offset = 442, length = 2,
            handle_func = function(input)
                return num_convert(input / 500)
            end
        },
        RXUpperThresholdCritical = {
            offset = 448, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        RXLowerThresholdCritical = {
            offset = 450, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        Power_TXUpperThresholdCritical = {
            offset = 432, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        },
        Power_TXLowerThresholdCritical = {
            offset = 434, length = 2,
            handle_func = function(input)
                return num_convert(input / 10000)
            end
        }
    },
    [IDENTIFIER_DEFAULT] = {
        ReadingCelsius = {
            default = 0xFFFF
        },
        SupplyVoltage = {
            default = 0xFFFF
        },
        TXBiasCurrentMilliAmps = {
            default = {0xFFFF}
        },
        RXInputPowerMilliWatts = {
            default = {0xFFFF}
        },
        TXOutputPowerMilliWatts = {
            default = {0xFFFF}
        },
        RxLossState = {
            default = false
        },
        TxLossState = {
            default = false
        },
        TxFaultState = {
            default = false
        },
        Temp_UpperThresholdCritical = {
            default = 0xFFFF
        },
        TemperatureUpperThresholdCritical = {
            default = 0xFFFF
        },
        Temp_LowerThresholdCritical = {
            default = 0xFFFF
        },
        TemperatureLowerThresholdCritical = {
            default = 0xFFFF
        },
        Vol_UpperThresholdCritical = {
            default = 0xFFFF
        },
        VoltageUpperThresholdCritical = {
            default = 0xFFFF
        },
        Vol_LowerThresholdCritical = {
            default = 0xFFFF
        },
        VoltageLowerThresholdCritical = {
            default = 0xFFFF
        },
        BC_TXUpperThresholdCritical = {
            default = 0xFFFF
        },
        BC_TXLowerThresholdCritical = {
            default = 0xFFFF
        },
        RXUpperThresholdCritical = {
            default = 0xFFFF
        },
        RXLowerThresholdCritical = {
            default = 0xFFFF
        },
        Power_TXUpperThresholdCritical = {
            default = 0xFFFF
        },
        Power_TXLowerThresholdCritical = {
            default = 0xFFFF
        }
    }
}

return OpticalModuleConfig