-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--          http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.
local mdb = require 'mc.mdb'
local class = require 'mc.class'
local app_base = require 'mc.client_app_base'
local mdb_service = require 'mc.mdb.mdb_service'
local subscribe_signal = require 'mc.mdb.subscribe_signal'
local org_freedesktop_dbus = require 'sd_bus.org_freedesktop_dbus'

local match_rule = org_freedesktop_dbus.MatchRule
local get_non_virtual_interface_objects = mdb_service.get_non_virtual_interface_objects
local foreach_non_virtual_interface_objects = mdb_service.foreach_non_virtual_interface_objects

local BlockIO = require 'network_adapter.json_types.BlockIO'
local PCIe = require 'network_adapter.json_types.PCIe'
local PCIeEndpoint = require 'network_adapter.json_types.PCIeEndpoint'
local PCIeTransport = require 'network_adapter.json_types.PCIeTransport'
local EthernetInterfaces = require 'network_adapter.json_types.EthernetInterfaces'
local MgmtPort = require 'network_adapter.json_types.MgmtPort'
local InnerNetwork = require 'network_adapter.json_types.InnerNetwork'
local LldpConfig = require 'network_adapter.json_types.LldpConfig'
local Frudata = require 'network_adapter.json_types.Frudata'
local NCSICapabilities = require 'network_adapter.json_types.NCSICapabilities'
local FruCtrl = require 'network_adapter.json_types.FruCtrl'
local Bios = require 'network_adapter.json_types.Bios'
local Accessor = require 'network_adapter.json_types.Accessor'
local DPUCard = require 'network_adapter.json_types.DPUCard'
local PCIeCard = require 'network_adapter.json_types.PCIeCard'
local SmsStatus = require 'network_adapter.json_types.SmsStatus'
local Sms = require 'network_adapter.json_types.Sms'
local NetworkBonding = require 'network_adapter.json_types.NetworkBonding'
local Events = require 'network_adapter.json_types.Events'
local Pmu = require 'network_adapter.json_types.Pmu'
local SmBios = require 'network_adapter.json_types.SmBios'

---@class network_adapter_client: BasicClient
local network_adapter_client = class(app_base.Client)

function network_adapter_client:GetBlockIOObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Chip.BlockIO', true)
end

function network_adapter_client:ForeachBlockIOObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Chip.BlockIO', cb, true)
end

function network_adapter_client:GetPCIeObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Mctp.MctpBinding.PCIe', true)
end

function network_adapter_client:ForeachPCIeObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Mctp.MctpBinding.PCIe', cb, true)
end

function network_adapter_client:GetPCIeEndpointObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Mctp.PCIeEndpoint', true)
end

function network_adapter_client:ForeachPCIeEndpointObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Mctp.PCIeEndpoint', cb, true)
end

function network_adapter_client:GetPCIeTransportObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Mctp.PCIeTransport', true)
end

function network_adapter_client:ForeachPCIeTransportObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Mctp.PCIeTransport', cb, true)
end

function network_adapter_client:GetEthernetInterfacesObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.EthernetInterfaces', true)
end

function network_adapter_client:ForeachEthernetInterfacesObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.EthernetInterfaces', cb, true)
end

function network_adapter_client:GetMgmtPortObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.EthernetInterfaces.MgmtPort', true)
end

function network_adapter_client:ForeachMgmtPortObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.EthernetInterfaces.MgmtPort', cb,
        true)
end

function network_adapter_client:GetInnerNetworkObjects()
    return
        get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.EthernetInterfaces.InnerNetwork', true)
end

function network_adapter_client:ForeachInnerNetworkObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.EthernetInterfaces.InnerNetwork',
        cb, true)
end

function network_adapter_client:GetLldpConfigObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.LldpConfig', true)
end

function network_adapter_client:ForeachLldpConfigObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.LldpConfig', cb, true)
end

function network_adapter_client:GetFrudataObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.FrudataService.Frudata', true)
end

function network_adapter_client:ForeachFrudataObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.FrudataService.Frudata', cb, true)
end

function network_adapter_client:GetNCSICapabilitiesObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.NCSICapabilities', true)
end

function network_adapter_client:ForeachNCSICapabilitiesObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.NCSICapabilities', cb, true)
end

function network_adapter_client:GetFruCtrlObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.FruCtrl', true)
end

function network_adapter_client:ForeachFruCtrlObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.FruCtrl', cb, true)
end

function network_adapter_client:GetAccessorObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Accessor', true)
end

function network_adapter_client:ForeachAccessorObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Accessor', cb, true)
end

function network_adapter_client:GetDPUCardObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.DPUCard', true)
end

function network_adapter_client:ForeachDPUCardObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.DPUCard', cb, true)
end

function network_adapter_client:GetPCIeCardObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.PCIeDevices.PCIeCard', true)
end

function network_adapter_client:ForeachPCIeCardObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.PCIeDevices.PCIeCard', cb, true)
end

function network_adapter_client:GetNetworkBondingObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.NetworkBonding', true)
end

function network_adapter_client:ForeachNetworkBondingObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.NetworkBonding', cb, true)
end

function network_adapter_client:GetEventsObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Events', true)
end

function network_adapter_client:ForeachEventsObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Events', cb, true)
end

function network_adapter_client:GetPmuObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Pmu', true)
end

function network_adapter_client:ForeachPmuObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Pmu', cb, true)
end

function network_adapter_client:GetSmBiosObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.SmBios', true)
end

function network_adapter_client:ForeachSmBiosObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.SmBios', cb, true)
end

function network_adapter_client:OnBlockIOInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chip.BlockIO')
end

function network_adapter_client:OnBlockIOInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chip.BlockIO')
end

function network_adapter_client:OnPCIeInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Mctp.MctpBinding.PCIe')
end

function network_adapter_client:OnPCIeInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Mctp.MctpBinding.PCIe')
end

function network_adapter_client:OnPCIeEndpointInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Mctp.PCIeEndpoint')
end

function network_adapter_client:OnPCIeEndpointInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Mctp.PCIeEndpoint')
end

function network_adapter_client:OnPCIeTransportInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Mctp.PCIeTransport')
end

function network_adapter_client:OnPCIeTransportInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Mctp.PCIeTransport')
end

function network_adapter_client:OnEthernetInterfacesPropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.EthernetInterfaces', {'VLANId'})
end

function network_adapter_client:OnEthernetInterfacesInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.EthernetInterfaces')
end

function network_adapter_client:OnEthernetInterfacesInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.EthernetInterfaces')
end

function network_adapter_client:OnMgmtPortInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.EthernetInterfaces.MgmtPort')
end

function network_adapter_client:OnMgmtPortInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.EthernetInterfaces.MgmtPort')
end

function network_adapter_client:OnInnerNetworkPropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.EthernetInterfaces.InnerNetwork', {'VLANId'})
end

function network_adapter_client:OnInnerNetworkInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.EthernetInterfaces.InnerNetwork')
end

function network_adapter_client:OnInnerNetworkInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.EthernetInterfaces.InnerNetwork')
end

function network_adapter_client:OnLldpConfigInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.LldpConfig')
end

function network_adapter_client:OnLldpConfigInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.LldpConfig')
end

function network_adapter_client:OnFrudataInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.FrudataService.Frudata')
end

function network_adapter_client:OnFrudataInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.FrudataService.Frudata')
end

function network_adapter_client:OnNCSICapabilitiesInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.NCSICapabilities')
end

function network_adapter_client:OnNCSICapabilitiesInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.NCSICapabilities')
end

function network_adapter_client:OnFruCtrlPropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.FruCtrl', {'PowerState', 'SysResetDetected'})
end

function network_adapter_client:OnFruCtrlInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.FruCtrl')
end

function network_adapter_client:OnFruCtrlInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.FruCtrl')
end

function network_adapter_client:GetBiosBiosObject(path_params)
    return mdb.try_get_object(self:get_bus(), path_params and
        ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Bios') or '/bmc/kepler/Systems/:SystemId/Bios',
        'bmc.kepler.Systems.Bios')
end

function network_adapter_client:OnBiosPropertiesChanged(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Bios') or
                               '/bmc/kepler/Systems'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Systems.Bios', {'SystemStartupState'})
end

function network_adapter_client:OnBiosInterfacesAdded(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Bios') or
                               '/bmc/kepler/Systems'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.Systems.Bios')
end

function network_adapter_client:OnBiosInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Bios') or
                               '/bmc/kepler/Systems'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Systems.Bios')
end

function network_adapter_client:OnAccessorInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Accessor')
end

function network_adapter_client:OnAccessorInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Accessor')
end

function network_adapter_client:OnDPUCardInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.DPUCard')
end

function network_adapter_client:OnDPUCardInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.DPUCard')
end

function network_adapter_client:OnPCIeCardInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.PCIeDevices.PCIeCard')
end

function network_adapter_client:OnPCIeCardInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.PCIeDevices.PCIeCard')
end

function network_adapter_client:GetSmsSmsStatusObject(path_params)
    return mdb.try_get_object(self:get_bus(), path_params and
        ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or '/bmc/kepler/Systems/${SystemId}/Sms',
        'bmc.kepler.Systems.Sms.SmsStatus')
end

function network_adapter_client:OnSmsStatusPropertiesChanged(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or
                               '/bmc/kepler/Systems'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Systems.Sms.SmsStatus', {'State'})
end

function network_adapter_client:OnSmsStatusInterfacesAdded(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or
                               '/bmc/kepler/Systems'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.Systems.Sms.SmsStatus')
end

function network_adapter_client:OnSmsStatusInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or
                               '/bmc/kepler/Systems'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Systems.Sms.SmsStatus')
end

function network_adapter_client:GetSmsSmsObject(path_params)
    return mdb.try_get_object(self:get_bus(), path_params and
        ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or '/bmc/kepler/Systems/${SystemId}/Sms',
        'bmc.kepler.Systems.Sms')
end

function network_adapter_client:OnSmsInterfacesAdded(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or
                               '/bmc/kepler/Systems'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.Systems.Sms')
end

function network_adapter_client:OnSmsInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or
                               '/bmc/kepler/Systems'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Systems.Sms')
end

function network_adapter_client:OnNetworkBondingInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.NetworkBonding')
end

function network_adapter_client:OnNetworkBondingInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.NetworkBonding')
end

function network_adapter_client:OnEventsInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Events')
end

function network_adapter_client:OnEventsInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Events')
end

function network_adapter_client:OnPmuPropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Pmu', {'Status', 'SFPMaxTemperature'})
end

function network_adapter_client:OnPmuInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Pmu')
end

function network_adapter_client:OnPmuInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Pmu')
end

function network_adapter_client:OnSmBiosPropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.SmBios', {'SmBiosStatus'})
end

function network_adapter_client:OnSmBiosInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.SmBios')
end

function network_adapter_client:OnSmBiosInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.SmBios')
end

---@return Bios.ClearCmosRsp
function network_adapter_client:BiosBiosClearCmos(ctx, path_params)
    local obj = self:GetBiosBiosObject(path_params)

    return Bios.ClearCmosRsp.new(obj:ClearCmos(ctx))
end

function network_adapter_client:PBiosBiosClearCmos(ctx, path_params)
    return pcall(function()
        local obj = self:GetBiosBiosObject(path_params)

        return Bios.ClearCmosRsp.new(obj:ClearCmos(ctx))
    end)
end

---@param Mode integer
---@return Bios.SetBiosConfigActiveModeRsp
function network_adapter_client:BiosBiosSetBiosConfigActiveMode(ctx, path_params, Mode)
    local req = Bios.SetBiosConfigActiveModeReq.new(Mode):validate()
    local obj = self:GetBiosBiosObject(path_params)

    return Bios.SetBiosConfigActiveModeRsp.new(obj:SetBiosConfigActiveMode(ctx, req:unpack(true)))
end

function network_adapter_client:PBiosBiosSetBiosConfigActiveMode(ctx, path_params, Mode)
    return pcall(function()
        local req = Bios.SetBiosConfigActiveModeReq.new(Mode):validate()
        local obj = self:GetBiosBiosObject(path_params)

        return Bios.SetBiosConfigActiveModeRsp.new(obj:SetBiosConfigActiveMode(ctx, req:unpack(true)))
    end)
end

---@return Bios.ResetSetupConfigRsp
function network_adapter_client:BiosBiosResetSetupConfig(ctx, path_params)
    local obj = self:GetBiosBiosObject(path_params)

    return Bios.ResetSetupConfigRsp.new(obj:ResetSetupConfig(ctx))
end

function network_adapter_client:PBiosBiosResetSetupConfig(ctx, path_params)
    return pcall(function()
        local obj = self:GetBiosBiosObject(path_params)

        return Bios.ResetSetupConfigRsp.new(obj:ResetSetupConfig(ctx))
    end)
end

---@param PasswordType string
---@param OldPassword string
---@param NewPassword string
---@return Bios.ChangePasswordRsp
function network_adapter_client:BiosBiosChangePassword(ctx, path_params, PasswordType, OldPassword, NewPassword)
    local req = Bios.ChangePasswordReq.new(PasswordType, OldPassword, NewPassword):validate()
    local obj = self:GetBiosBiosObject(path_params)

    return Bios.ChangePasswordRsp.new(obj:ChangePassword(ctx, req:unpack(true)))
end

function network_adapter_client:PBiosBiosChangePassword(ctx, path_params, PasswordType, OldPassword, NewPassword)
    return pcall(function()
        local req = Bios.ChangePasswordReq.new(PasswordType, OldPassword, NewPassword):validate()
        local obj = self:GetBiosBiosObject(path_params)

        return Bios.ChangePasswordRsp.new(obj:ChangePassword(ctx, req:unpack(true)))
    end)
end

---@param BiosLogPrintEnabled integer
---@return Bios.SetBiosLogEnabledRsp
function network_adapter_client:BiosBiosSetBiosLogEnabled(ctx, path_params, BiosLogPrintEnabled)
    local req = Bios.SetBiosLogEnabledReq.new(BiosLogPrintEnabled):validate()
    local obj = self:GetBiosBiosObject(path_params)

    return Bios.SetBiosLogEnabledRsp.new(obj:SetBiosLogEnabled(ctx, req:unpack(true)))
end

function network_adapter_client:PBiosBiosSetBiosLogEnabled(ctx, path_params, BiosLogPrintEnabled)
    return pcall(function()
        local req = Bios.SetBiosLogEnabledReq.new(BiosLogPrintEnabled):validate()
        local obj = self:GetBiosBiosObject(path_params)

        return Bios.SetBiosLogEnabledRsp.new(obj:SetBiosLogEnabled(ctx, req:unpack(true)))
    end)
end

---@param FileType string
---@return Bios.ExportBiosSetupRsp
function network_adapter_client:BiosBiosExportBiosSetup(ctx, path_params, FileType)
    local req = Bios.ExportBiosSetupReq.new(FileType):validate()
    local obj = self:GetBiosBiosObject(path_params)

    return Bios.ExportBiosSetupRsp.new(obj:ExportBiosSetup(ctx, req:unpack(true)))
end

function network_adapter_client:PBiosBiosExportBiosSetup(ctx, path_params, FileType)
    return pcall(function()
        local req = Bios.ExportBiosSetupReq.new(FileType):validate()
        local obj = self:GetBiosBiosObject(path_params)

        return Bios.ExportBiosSetupRsp.new(obj:ExportBiosSetup(ctx, req:unpack(true)))
    end)
end

---@param FileType string
---@param Content string
---@return Bios.ImportBiosSetupRsp
function network_adapter_client:BiosBiosImportBiosSetup(ctx, path_params, FileType, Content)
    local req = Bios.ImportBiosSetupReq.new(FileType, Content):validate()
    local obj = self:GetBiosBiosObject(path_params)

    return Bios.ImportBiosSetupRsp.new(obj:ImportBiosSetup(ctx, req:unpack(true)))
end

function network_adapter_client:PBiosBiosImportBiosSetup(ctx, path_params, FileType, Content)
    return pcall(function()
        local req = Bios.ImportBiosSetupReq.new(FileType, Content):validate()
        local obj = self:GetBiosBiosObject(path_params)

        return Bios.ImportBiosSetupRsp.new(obj:ImportBiosSetup(ctx, req:unpack(true)))
    end)
end

---@param ReqKey string
---@return Bios.GetCurrentValueSettingRsp
function network_adapter_client:BiosBiosGetCurrentValueSetting(ctx, path_params, ReqKey)
    local req = Bios.GetCurrentValueSettingReq.new(ReqKey):validate()
    local obj = self:GetBiosBiosObject(path_params)

    return Bios.GetCurrentValueSettingRsp.new(obj:GetCurrentValueSetting(ctx, req:unpack(true)))
end

function network_adapter_client:PBiosBiosGetCurrentValueSetting(ctx, path_params, ReqKey)
    return pcall(function()
        local req = Bios.GetCurrentValueSettingReq.new(ReqKey):validate()
        local obj = self:GetBiosBiosObject(path_params)

        return Bios.GetCurrentValueSettingRsp.new(obj:GetCurrentValueSetting(ctx, req:unpack(true)))
    end)
end

---@return Bios.RemoveCachedSettingsRsp
function network_adapter_client:BiosBiosRemoveCachedSettings(ctx, path_params)
    local obj = self:GetBiosBiosObject(path_params)

    return Bios.RemoveCachedSettingsRsp.new(obj:RemoveCachedSettings(ctx))
end

function network_adapter_client:PBiosBiosRemoveCachedSettings(ctx, path_params)
    return pcall(function()
        local obj = self:GetBiosBiosObject(path_params)

        return Bios.RemoveCachedSettingsRsp.new(obj:RemoveCachedSettings(ctx))
    end)
end

---@param Request string
---@return Sms.SetForwardWhiteListRsp
function network_adapter_client:SmsSmsSetForwardWhiteList(ctx, path_params, Request)
    local req = Sms.SetForwardWhiteListReq.new(Request):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.SetForwardWhiteListRsp.new(obj:SetForwardWhiteList(ctx, req:unpack(true)))
end

function network_adapter_client:PSmsSmsSetForwardWhiteList(ctx, path_params, Request)
    return pcall(function()
        local req = Sms.SetForwardWhiteListReq.new(Request):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.SetForwardWhiteListRsp.new(obj:SetForwardWhiteList(ctx, req:unpack(true)))
    end)
end

---@param Request string
---@param RetryTimes integer
---@return Sms.ForwardRequestRsp
function network_adapter_client:SmsSmsForwardRequest(ctx, path_params, Request, RetryTimes)
    local req = Sms.ForwardRequestReq.new(Request, RetryTimes):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.ForwardRequestRsp.new(obj:ForwardRequest(ctx, req:unpack(true)))
end

function network_adapter_client:PSmsSmsForwardRequest(ctx, path_params, Request, RetryTimes)
    return pcall(function()
        local req = Sms.ForwardRequestReq.new(Request, RetryTimes):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.ForwardRequestRsp.new(obj:ForwardRequest(ctx, req:unpack(true)))
    end)
end

---@param RequestMethod string
---@param RequestToken string
---@param RequestBody string
---@return Sms.ForwardEventRsp
function network_adapter_client:SmsSmsForwardEvent(ctx, path_params, RequestMethod, RequestToken, RequestBody)
    local req = Sms.ForwardEventReq.new(RequestMethod, RequestToken, RequestBody):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.ForwardEventRsp.new(obj:ForwardEvent(ctx, req:unpack(true)))
end

function network_adapter_client:PSmsSmsForwardEvent(ctx, path_params, RequestMethod, RequestToken, RequestBody)
    return pcall(function()
        local req = Sms.ForwardEventReq.new(RequestMethod, RequestToken, RequestBody):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.ForwardEventRsp.new(obj:ForwardEvent(ctx, req:unpack(true)))
    end)
end

---@param ResourceType string
---@return Sms.UpdateSpecificResourceRsp
function network_adapter_client:SmsSmsUpdateSpecificResource(ctx, path_params, ResourceType)
    local req = Sms.UpdateSpecificResourceReq.new(ResourceType):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.UpdateSpecificResourceRsp.new(obj:UpdateSpecificResource(ctx, req:unpack(true)))
end

function network_adapter_client:PSmsSmsUpdateSpecificResource(ctx, path_params, ResourceType)
    return pcall(function()
        local req = Sms.UpdateSpecificResourceReq.new(ResourceType):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.UpdateSpecificResourceRsp.new(obj:UpdateSpecificResource(ctx, req:unpack(true)))
    end)
end

---@param ChannelType integer
---@param IsPowerOffPer boolean
---@return Sms.SetChannelTypeRsp
function network_adapter_client:SmsSmsSetChannelType(ctx, path_params, ChannelType, IsPowerOffPer)
    local req = Sms.SetChannelTypeReq.new(ChannelType, IsPowerOffPer):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.SetChannelTypeRsp.new(obj:SetChannelType(ctx, req:unpack(true)))
end

function network_adapter_client:PSmsSmsSetChannelType(ctx, path_params, ChannelType, IsPowerOffPer)
    return pcall(function()
        local req = Sms.SetChannelTypeReq.new(ChannelType, IsPowerOffPer):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.SetChannelTypeRsp.new(obj:SetChannelType(ctx, req:unpack(true)))
    end)
end

---@param IsPowerOffPer boolean
---@return Sms.GetChannelTypeRsp
function network_adapter_client:SmsSmsGetChannelType(ctx, path_params, IsPowerOffPer)
    local req = Sms.GetChannelTypeReq.new(IsPowerOffPer):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.GetChannelTypeRsp.new(obj:GetChannelType(ctx, req:unpack(true)))
end

function network_adapter_client:PSmsSmsGetChannelType(ctx, path_params, IsPowerOffPer)
    return pcall(function()
        local req = Sms.GetChannelTypeReq.new(IsPowerOffPer):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.GetChannelTypeRsp.new(obj:GetChannelType(ctx, req:unpack(true)))
    end)
end

---@param Enabled boolean
---@return Sms.SetCdevChannelEnabledRsp
function network_adapter_client:SmsSmsSetCdevChannelEnabled(ctx, path_params, Enabled)
    local req = Sms.SetCdevChannelEnabledReq.new(Enabled):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.SetCdevChannelEnabledRsp.new(obj:SetCdevChannelEnabled(ctx, req:unpack(true)))
end

function network_adapter_client:PSmsSmsSetCdevChannelEnabled(ctx, path_params, Enabled)
    return pcall(function()
        local req = Sms.SetCdevChannelEnabledReq.new(Enabled):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.SetCdevChannelEnabledRsp.new(obj:SetCdevChannelEnabled(ctx, req:unpack(true)))
    end)
end

function network_adapter_client:SubscribePCIeEndpointOnMessage(cb)
    local sig = match_rule.signal('OnMessage', 'bmc.kepler.Systems.Mctp.PCIeEndpoint')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function network_adapter_client:SubscribePCIeEndpointMessageReceived(cb)
    local sig = match_rule.signal('MessageReceived', 'bmc.kepler.Systems.Mctp.PCIeEndpoint')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function network_adapter_client:SubscribeEthernetInterfacesActivePortChangedSignal(cb)
    local sig = match_rule.signal('ActivePortChangedSignal', 'bmc.kepler.Managers.EthernetInterfaces')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function network_adapter_client:SubscribeEthernetInterfacesNCSIInfoChangedSignal(cb)
    local sig = match_rule.signal('NCSIInfoChangedSignal', 'bmc.kepler.Managers.EthernetInterfaces')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function network_adapter_client:SubscribeEthernetInterfacesEthMacChangedSignal(cb)
    local sig = match_rule.signal('EthMacChangedSignal', 'bmc.kepler.Managers.EthernetInterfaces')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function network_adapter_client:SubscribeLldpConfigLLDPOverNCSIStateChanged(cb)
    local sig = match_rule.signal('LLDPOverNCSIStateChanged', 'bmc.kepler.Managers.LldpConfig')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function network_adapter_client:SubscribeFruCtrlBeforePowerOnSignal(cb)
    local sig = match_rule.signal('BeforePowerOnSignal', 'bmc.kepler.Systems.FruCtrl')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function network_adapter_client:ctor()
    self.signal_slots = {}
end

---@type network_adapter_client
return network_adapter_client.new('network_adapter')
