-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local validate = require 'mc.validate'
local utils = require 'mc.utils'

local msg = {}
local defs = {}

---@class defs.CertInfoStruct
---@field Issuer string
---@field Subject string
---@field ValidNotBefore string
---@field ValidNotAfter string
---@field SerialNumber string
---@field SignatureAlgorithm string
---@field KeyUsage string
---@field PublicKeyLengthBits integer
local TCertInfoStruct = {}
TCertInfoStruct.__index = TCertInfoStruct
TCertInfoStruct.group = {}

local function TCertInfoStruct_from_obj(obj)
    return setmetatable(obj, TCertInfoStruct)
end

function TCertInfoStruct.new(Issuer, Subject, ValidNotBefore, ValidNotAfter, SerialNumber,
    SignatureAlgorithm, KeyUsage, PublicKeyLengthBits)
    return TCertInfoStruct_from_obj({
        Issuer = Issuer,
        Subject = Subject,
        ValidNotBefore = ValidNotBefore,
        ValidNotAfter = ValidNotAfter,
        SerialNumber = SerialNumber,
        SignatureAlgorithm = SignatureAlgorithm,
        KeyUsage = KeyUsage,
        PublicKeyLengthBits = PublicKeyLengthBits
    })
end
---@param obj defs.CertInfoStruct
function TCertInfoStruct:init_from_obj(obj)
    self.Issuer = obj.Issuer
    self.Subject = obj.Subject
    self.ValidNotBefore = obj.ValidNotBefore
    self.ValidNotAfter = obj.ValidNotAfter
    self.SerialNumber = obj.SerialNumber
    self.SignatureAlgorithm = obj.SignatureAlgorithm
    self.KeyUsage = obj.KeyUsage
    self.PublicKeyLengthBits = obj.PublicKeyLengthBits
end

function TCertInfoStruct:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TCertInfoStruct.group)
end

TCertInfoStruct.from_obj = TCertInfoStruct_from_obj

TCertInfoStruct.proto_property = {
    'Issuer', 'Subject', 'ValidNotBefore', 'ValidNotAfter', 'SerialNumber', 'SignatureAlgorithm',
    'KeyUsage', 'PublicKeyLengthBits'
}

TCertInfoStruct.default = {'', '', '', '', '', '', '', 0}

TCertInfoStruct.struct = {
    {name = 'Issuer', is_array = false, struct = nil},
    {name = 'Subject', is_array = false, struct = nil},
    {name = 'ValidNotBefore', is_array = false, struct = nil},
    {name = 'ValidNotAfter', is_array = false, struct = nil},
    {name = 'SerialNumber', is_array = false, struct = nil},
    {name = 'SignatureAlgorithm', is_array = false, struct = nil},
    {name = 'KeyUsage', is_array = false, struct = nil},
    {name = 'PublicKeyLengthBits', is_array = false, struct = nil}
}

function TCertInfoStruct:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'Issuer', self.Issuer, 'string', false, errs)
    validate.Required(prefix .. 'Subject', self.Subject, 'string', false, errs)
    validate.Required(prefix .. 'ValidNotBefore', self.ValidNotBefore, 'string', false, errs)
    validate.Required(prefix .. 'ValidNotAfter', self.ValidNotAfter, 'string', false, errs)
    validate.Required(prefix .. 'SerialNumber', self.SerialNumber, 'string', false, errs)
    validate.Required(prefix .. 'SignatureAlgorithm', self.SignatureAlgorithm, 'string', false, errs)
    validate.Required(prefix .. 'KeyUsage', self.KeyUsage, 'string', false, errs)
    validate.Required(prefix .. 'PublicKeyLengthBits', self.PublicKeyLengthBits, 'uint32', false,
        errs)

    TCertInfoStruct:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TCertInfoStruct.proto_property, errs)
    return self
end

function TCertInfoStruct:unpack(_)
    return self.Issuer, self.Subject, self.ValidNotBefore, self.ValidNotAfter, self.SerialNumber,
        self.SignatureAlgorithm, self.KeyUsage, self.PublicKeyLengthBits
end

defs.CertInfoStruct = TCertInfoStruct

---@class Smtp.GetCertInfoRsp
---@field CertInfo defs.CertInfoStruct
local TGetCertInfoRsp = {}
TGetCertInfoRsp.__index = TGetCertInfoRsp
TGetCertInfoRsp.group = {}

local function TGetCertInfoRsp_from_obj(obj)
    obj.CertInfo = utils.from_obj(defs.CertInfoStruct, obj.CertInfo)
    return setmetatable(obj, TGetCertInfoRsp)
end

function TGetCertInfoRsp.new(CertInfo)
    return TGetCertInfoRsp_from_obj({CertInfo = CertInfo})
end
---@param obj Smtp.GetCertInfoRsp
function TGetCertInfoRsp:init_from_obj(obj)
    self.CertInfo = obj.CertInfo
end

function TGetCertInfoRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TGetCertInfoRsp.group)
end

TGetCertInfoRsp.from_obj = TGetCertInfoRsp_from_obj

TGetCertInfoRsp.proto_property = {'CertInfo'}

TGetCertInfoRsp.default = {defs.CertInfoStruct.default}

TGetCertInfoRsp.struct = {
    {name = 'CertInfo', is_array = false, struct = defs.CertInfoStruct.struct}
}

function TGetCertInfoRsp:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'CertInfo', self.CertInfo, 'defs.CertInfoStruct', false, errs)

    TGetCertInfoRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TGetCertInfoRsp.proto_property, errs)
    return self
end

function TGetCertInfoRsp:unpack(raw)
    return utils.unpack(raw, self.CertInfo)
end

msg.GetCertInfoRsp = TGetCertInfoRsp

---@class Smtp.GetCertInfoReq
local TGetCertInfoReq = {}
TGetCertInfoReq.__index = TGetCertInfoReq
TGetCertInfoReq.group = {}

local function TGetCertInfoReq_from_obj(obj)
    return setmetatable(obj, TGetCertInfoReq)
end

function TGetCertInfoReq.new()
    return TGetCertInfoReq_from_obj({})
end
---@param obj Smtp.GetCertInfoReq
function TGetCertInfoReq:init_from_obj(obj)

end

function TGetCertInfoReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TGetCertInfoReq.group)
end

TGetCertInfoReq.from_obj = TGetCertInfoReq_from_obj

TGetCertInfoReq.proto_property = {}

TGetCertInfoReq.default = {}

TGetCertInfoReq.struct = {}

function TGetCertInfoReq:validate(prefix, errs)
    prefix = prefix or ''

    TGetCertInfoReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TGetCertInfoReq.proto_property, errs)
    return self
end

function TGetCertInfoReq:unpack(_)
end

msg.GetCertInfoReq = TGetCertInfoReq

---@class Smtp.ImportCertRsp
---@field TaskId integer
local TImportCertRsp = {}
TImportCertRsp.__index = TImportCertRsp
TImportCertRsp.group = {}

local function TImportCertRsp_from_obj(obj)
    return setmetatable(obj, TImportCertRsp)
end

function TImportCertRsp.new(TaskId)
    return TImportCertRsp_from_obj({TaskId = TaskId})
end
---@param obj Smtp.ImportCertRsp
function TImportCertRsp:init_from_obj(obj)
    self.TaskId = obj.TaskId
end

function TImportCertRsp:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TImportCertRsp.group)
end

TImportCertRsp.from_obj = TImportCertRsp_from_obj

TImportCertRsp.proto_property = {'TaskId'}

TImportCertRsp.default = {0}

TImportCertRsp.struct = {{name = 'TaskId', is_array = false, struct = nil}}

function TImportCertRsp:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'TaskId', self.TaskId, 'uint32', false, errs)

    TImportCertRsp:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TImportCertRsp.proto_property, errs)
    return self
end

function TImportCertRsp:unpack(_)
    return self.TaskId
end

msg.ImportCertRsp = TImportCertRsp

---@class Smtp.ImportCertReq
---@field FilePath string
local TImportCertReq = {}
TImportCertReq.__index = TImportCertReq
TImportCertReq.group = {}

local function TImportCertReq_from_obj(obj)
    return setmetatable(obj, TImportCertReq)
end

function TImportCertReq.new(FilePath)
    return TImportCertReq_from_obj({FilePath = FilePath})
end
---@param obj Smtp.ImportCertReq
function TImportCertReq:init_from_obj(obj)
    self.FilePath = obj.FilePath
end

function TImportCertReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TImportCertReq.group)
end

TImportCertReq.from_obj = TImportCertReq_from_obj

TImportCertReq.proto_property = {'FilePath'}

TImportCertReq.default = {''}

TImportCertReq.struct = {{name = 'FilePath', is_array = false, struct = nil}}

function TImportCertReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'FilePath', self.FilePath, 'string', false, errs)

    TImportCertReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TImportCertReq.proto_property, errs)
    return self
end

function TImportCertReq:unpack(_)
    return self.FilePath
end

msg.ImportCertReq = TImportCertReq

local Smtp = {}

local Mdb = require 'mc.mdb'

---@class Smtp: Table
---@field Enabled boolean
---@field Address string
---@field Port integer
---@field TLSEnabled boolean
---@field ServerAuthEnabled boolean
---@field LowestSeverity string
Smtp.Smtp = Mdb.register_interface('bmc.kepler.EventService.Subscriptions.Smtp', {
    Enabled = {'b', nil, false, False},
    Address = {'s', nil, false, nil},
    Port = {'q', nil, false, 25},
    TLSEnabled = {'b', nil, false, True},
    ServerAuthEnabled = {'b', nil, false, False},
    LowestSeverity = {'s', nil, false, 'Normal'}
}, {
    ImportCert = {'a{ss}s', 'u', msg.ImportCertReq, msg.ImportCertRsp},
    GetCertInfo = {'a{ss}', '(sssssssu)', msg.GetCertInfoReq, msg.GetCertInfoRsp}
}, {})
return Smtp
