-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local lu = require 'luaunit'
local get_version = require 'external_interface.function_get_version'
local client = require 'manufacture.client'
local manufacture_service = require 'manufacture_service'
local ipmi = require 'manufacture.ipmi.cmds.GetTestItemCount'
local mc_ipmi = require 'ipmi'
local ipmi_get_test_item_count = require 'ipmi_cmd.get_test_item_count'
local factory = require 'factory'

Test = {}

function Test:setupClass()
    self.GetFirmwareInfoObjects = client.GetFirmwareInfoObjects
    self.GetBoardObjects = client.GetBoardObjects
end

function Test:teardownClass()
    client.GetFirmwareInfoObjects = Test.GetFirmwareInfoObjects
    client.GetBoardObjects = Test.GetBoardObjects
end

function Test:test_get_vrd_version_by_fruid()
    client.GetFirmwareInfoObjects = function ()
        return {
            {
                Id = "VRD_BCU",
                Name = "CpuBoard1 VRD",
                Version = "29.29.29.29.29.29.29.29"
            }
        }
    end
    client.GetBoardObjects = function ()
        return {
            {
                FruID = 4,
                DeviceName = "CpuBoard1",
                Version = "29.29.29.29.29.29.29.29"
            }
        }
    end
    local version = get_version.get_version[0x1a](4)
    lu.assertEquals(version, "29.29.29.29.29.29.29.29")
end

function Test:test_get_psip_version_by_fruid()
    client.GetBoardObjects = function ()
        return {
            {
                FruID = 4,
                DeviceName = "CpuBoard1",
                PSIPVersion = "1111"
            }
        }
    end
    local version = get_version.get_version[0x1C](4)
    lu.assertEquals(version, "1111")
end

function Test:test_get_bios_version_when_hostid_is_one()
    client.GetFirmwareInfoObjects = function ()
        return {
            {
                Id = "Bios",
                Version = "6.59",
                Location = "U75"
            }
        }
    end
    local version = get_version.get_version[0x06](0)
    lu.assertEquals(version, "(U75)6.59")
    version = get_version.get_version[0x06](0, 1)
    lu.assertEquals(version, "(U75)6.59")
end

function Test:test_get_bios_version_when_hostid_not_one()
    client.GetFirmwareInfoObjects = function ()
        return {
            {
                Id = "Bios2",
                Version = "7.00",
                Location = "U69"
            }
        }
    end
    local version = get_version.get_version[0x06](0, 2)
    lu.assertEquals(version, "(U69)7.00")
    version = get_version.get_version[0x06]()
    lu.assertEquals(version, "")
    version = get_version.get_version[0x06](0, 1)
    lu.assertEquals(version, "")
end

function Test:test_get_bios_version_when_no_objects()
    client.GetFirmwareInfoObjects = function ()
        return {
            {}
        }
    end
    local version = get_version.get_version[0x06](0, 1)
    lu.assertEquals(version, "")
end

function Test:test_get_mulhost_test_count()
    client.GetFirmwareInfoObjects = function ()
        return {
            {}
        }
    end
    manufacture_service.insert_dft_test_path({
        host_type = {path = {SystemId = 1}},
        obj_list = {path = {Type = 1, Id = 1}}
    })
    lu.assertEquals(0, 0)
end

function Test:test_get_test_item_count()
    local req = {Type = 0x01}
    local ctx = {HostId = 0}
    local manufacture_service = manufacture_service.new()
    manufacture_service.get_type_total = function(type, id)
        if not type then 
            return type
        end
        return id
    end
    factory.register_to_factory("manufacture_service", manufacture_service)
    local obj = factory.get_obj("manufacture_service")
    lu.assertNotEquals(obj, nil)
    local ret = ipmi_get_test_item_count.get_test_item_count(req, ctx)
    local comp_code = mc_ipmi.types.Cc
    lu.assertEquals(ret.CompletionCode, comp_code.Success)
end