-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--          http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.
local Databases = require 'database'
local Col = require 'database.column'

local def_types = require 'class.types.types'

---@class SessionServiceTable: Table
---@field SessionTimeout FieldBase
---@field SessionMode FieldBase
---@field SessionMaxCount FieldBase
---@field SessionType FieldBase
---@field SessionModeDB FieldBase

---@class AuthenticationTable: Table
---@field AccountLockoutDuration FieldBase
---@field AccountLockoutThreshold FieldBase
---@field AccountLockoutCounterResetAfter FieldBase
---@field LocalAccountAuth FieldBase
---@field DoubleCertificationEnable FieldBase
---@field CertificateAuthenticationEnable FieldBase
---@field Id FieldBase

---@class LDAPTable: Table
---@field Enabled FieldBase
---@field Id FieldBase

---@class LDAPControllerTable: Table
---@field Enabled FieldBase
---@field HostAddr FieldBase
---@field Port FieldBase
---@field UserDomain FieldBase
---@field Folder FieldBase
---@field BindDN FieldBase
---@field CertVerifyEnabled FieldBase
---@field CertVerifyLevel FieldBase
---@field Id FieldBase
---@field LdapVer FieldBase
---@field BaseDN FieldBase
---@field BindDNPsw FieldBase
---@field Scope FieldBase
---@field TimeLimit FieldBase
---@field BindTimeLimit FieldBase
---@field CRLVerificationEnable FieldBase

---@class KerberosTable: Table
---@field Enabled FieldBase
---@field Address FieldBase
---@field Port FieldBase
---@field Realm FieldBase
---@field Id FieldBase

---@class RemoteGroupTable: Table
---@field GroupType FieldBase
---@field ControllerId FieldBase
---@field SID FieldBase
---@field Name FieldBase
---@field UserRoleId FieldBase
---@field Folder FieldBase
---@field PermitRuleIds FieldBase
---@field LoginInterface FieldBase
---@field Id FieldBase
---@field ControllerInnerId FieldBase
---@field Domain FieldBase
---@field Privilege FieldBase
---@field PrivilegeMask FieldBase
---@field PermitRuleIdsDB FieldBase
---@field LoginInterfaceDB FieldBase

---@class CertificateAuthenticationTable: Table
---@field Enabled FieldBase
---@field OCSPEnabled FieldBase
---@field Id FieldBase

---@class IamDBDatabase
---@field db DataBase
---@field select fun(db:DataBase, table: any, ...): SelectStatement
---@field update fun(db:DataBase, table: any, ...): UpdateStatement
---@field insert fun(db:DataBase, table: any, ...): InsertStatement
---@field SessionService SessionServiceTable
---@field Authentication AuthenticationTable
---@field LDAP LDAPTable
---@field LDAPController LDAPControllerTable
---@field Kerberos KerberosTable
---@field RemoteGroup RemoteGroupTable
---@field CertificateAuthentication CertificateAuthenticationTable
local IamDBDatabase = {}
IamDBDatabase.__index = IamDBDatabase

function IamDBDatabase.new(path, datas)
    local db = Databases(path)
    local obj = {db = db}

    obj.SessionService = db:Table('t_session_service', {
        SessionTimeout = Col.IntegerField():cid(1):persistence_key('protect_power_off'):null():max_length(32),
        SessionMode = Col.IntegerField():cid(2):null():max_length(32):default(1),
        SessionMaxCount = Col.IntegerField():cid(3):persistence_key('protect_power_off'):null():max_length(32),
        SessionType = Col.EnumField(def_types.SessionType):cid(4):primary_key():persistence_key('protect_power_off'),
        SessionModeDB = Col.EnumField(def_types.OccupationMode):cid(5):persistence_key('protect_power_off'):null()
    }):create_if_not_exist(datas and datas['t_session_service'])
    obj.Authentication = db:Table('t_authentication', {
        AccountLockoutDuration = Col.IntegerField():cid(1):persistence_key('protect_power_off'):null():max_length(32)
            :default(300),
        AccountLockoutThreshold = Col.IntegerField():cid(2):persistence_key('protect_power_off'):null():max_length(32)
            :default(5),
        AccountLockoutCounterResetAfter = Col.IntegerField():cid(3):persistence_key('protect_power_off'):null()
            :max_length(32):default(300),
        LocalAccountAuth = Col.TextField():cid(4):persistence_key('protect_power_off'):null():default('\'LocalFirst\''),
        DoubleCertificationEnable = Col.BooleandField():cid(5):persistence_key('protect_power_off'):null()
            :default(false),
        CertificateAuthenticationEnable = Col.BooleandField():cid(6):persistence_key('protect_power_off'):null()
            :default(false),
        Id = Col.IntegerField():cid(7):primary_key():persistence_key('protect_power_off'):max_length(8)
    }):create_if_not_exist(datas and datas['t_authentication'])
    obj.LDAP = db:Table('t_ldap', {
        Enabled = Col.BooleandField():cid(1):persistence_key('protect_power_off'):null():default(true),
        Id = Col.IntegerField():cid(2):primary_key():persistence_key('protect_power_off'):max_length(32)
    }):create_if_not_exist(datas and datas['t_ldap'])
    obj.LDAPController = db:Table('t_ldap_controller', {
        Enabled = Col.BooleandField():cid(1):persistence_key('protect_power_off'):null():default(true),
        HostAddr = Col.TextField():cid(2):persistence_key('protect_power_off'),
        Port = Col.IntegerField():cid(3):persistence_key('protect_power_off'):null():max_length(32):default(636),
        UserDomain = Col.TextField():cid(4):persistence_key('protect_power_off'),
        Folder = Col.TextField():cid(5):persistence_key('protect_power_off'),
        BindDN = Col.TextField():cid(6):persistence_key('protect_power_off'),
        CertVerifyEnabled = Col.BooleandField():cid(7):persistence_key('protect_power_off'):null():default(true),
        CertVerifyLevel = Col.IntegerField():cid(8):persistence_key('protect_power_off'):null():max_length(8):default(2),
        Id = Col.IntegerField():cid(9):primary_key():persistence_key('protect_power_off'):max_length(32),
        LdapVer = Col.IntegerField():cid(10):persistence_key('protect_power_off'):null():max_length(8):default(3),
        BaseDN = Col.TextField():cid(11):persistence_key('protect_power_off'),
        BindDNPsw = Col.TextField():cid(12):persistence_key('protect_power_off'),
        Scope = Col.TextField():cid(13):persistence_key('protect_power_off'):null():default('\'sub\''),
        TimeLimit = Col.IntegerField():cid(14):persistence_key('protect_power_off'):null():max_length(32):default(2),
        BindTimeLimit = Col.IntegerField():cid(15):persistence_key('protect_power_off'):null():max_length(32):default(2),
        CRLVerificationEnable = Col.BooleandField():cid(16):persistence_key('protect_power_off'):null():default(false)
    }):create_if_not_exist(datas and datas['t_ldap_controller'])
    obj.Kerberos = db:Table('t_kerberos', {
        Enabled = Col.BooleandField():cid(1):persistence_key('protect_power_off'):null():default(true),
        Address = Col.TextField():cid(2):persistence_key('protect_power_off'):null(),
        Port = Col.IntegerField():cid(3):persistence_key('protect_power_off'):null():max_length(32),
        Realm = Col.TextField():cid(4):persistence_key('protect_power_off'):null(),
        Id = Col.IntegerField():cid(5):primary_key():persistence_key('protect_power_off'):max_length(8)
    }):create_if_not_exist(datas and datas['t_kerberos'])
    obj.RemoteGroup = db:Table('t_remote_group', {
        GroupType = Col.IntegerField():cid(1):persistence_key('protect_power_off'):null():max_length(8),
        ControllerId = Col.IntegerField():cid(2):persistence_key('protect_power_off'):null():max_length(8),
        SID = Col.TextField():cid(3):persistence_key('protect_power_off'),
        Name = Col.TextField():cid(4):persistence_key('protect_power_off'),
        UserRoleId = Col.IntegerField():cid(5):persistence_key('protect_power_off'):max_length(8),
        Folder = Col.TextField():cid(6):persistence_key('protect_power_off'):null(),
        PermitRuleIds = Col.JsonField():cid(7):null():default([['[]']]),
        LoginInterface = Col.JsonField():cid(8):null():default([['[]']]),
        Id = Col.IntegerField():cid(9):primary_key():persistence_key('protect_power_off'):max_length(32),
        ControllerInnerId = Col.IntegerField():cid(10):primary_key():persistence_key('protect_power_off'):max_length(32),
        Domain = Col.TextField():cid(11):persistence_key('protect_power_off'):null(),
        Privilege = Col.IntegerField():cid(12):persistence_key('protect_power_off'):null():max_length(8),
        PrivilegeMask = Col.IntegerField():cid(13):persistence_key('protect_power_off'):null():max_length(32),
        PermitRuleIdsDB = Col.IntegerField():cid(14):persistence_key('protect_power_off'):null():max_length(8),
        LoginInterfaceDB = Col.IntegerField():cid(15):persistence_key('protect_power_off'):null():max_length(8)
    }):create_if_not_exist(datas and datas['t_remote_group'])
    obj.CertificateAuthentication = db:Table('t_certificate_authentication', {
        Enabled = Col.BooleandField():cid(1):persistence_key('protect_power_off'):null():default(false),
        OCSPEnabled = Col.BooleandField():cid(2):persistence_key('protect_power_off'):null():default(false),
        Id = Col.IntegerField():cid(3):primary_key():persistence_key('protect_power_off'):max_length(8)
    }):create_if_not_exist(datas and datas['t_certificate_authentication'])

    obj.tables = db.tables
    return setmetatable(obj, IamDBDatabase)
end

function IamDBDatabase:select(table, ...)
    return self.db:select(table, ...)
end

function IamDBDatabase:update(table, ...)
    return self.db:update(table, ...)
end

function IamDBDatabase:insert(table, ...)
    return self.db:insert(table, ...)
end

function IamDBDatabase:delete(table, ...)
    return self.db:delete(table, ...)
end

function IamDBDatabase:exec(...)
    return self.db:exec(...)
end

return IamDBDatabase.new
