-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

-- Description: retimer升级
local singleton = require 'mc.singleton'
local log = require 'mc.logging'
local cmn = require 'common'
local fw_mgmt = require 'general_hardware.client'
local ctx = require 'mc.context'
local parser_cfg = require 'retimer.upgrade.parser_cfg'
local RETIMER_CONSTANTS = require 'retimer.retimer_constants'
local utils = require 'mc.utils'
local upgrade_subject = require 'upgrade_subject.upgrade_subject'

local retimer_upg_service = {}
retimer_upg_service.__index = retimer_upg_service

local RET_OK<const> = 0
local RET_ERR<const> = -1

-- 升级主任务
--- @param system_id number 系统id
--- @param firmware_type string 升级类型
--- @param file_path string 升级文件目录/dev/shm/upgrade/FirmwareX
function retimer_upg_service:upgrade_task(system_id, firmware_type, file_path)
    -- 解析出升级目录和文件名
    local dir, firmware_name = parser_cfg.parse_dir(file_path)

    -- 判断固件属于哪种芯片
    local firmware_chip_type = self.cfgs[firmware_name].component_id_ex
    local chip_callback = self.register_chips_callback[firmware_chip_type]
    if chip_callback == nil then
        log:error('[RetimerUpgrade] Has no register chiptype %s.', firmware_chip_type)
        fw_mgmt:UpdateServiceUpdateServiceProcessReply(ctx.new(), system_id, firmware_type, RET_ERR)
        return
    end
    
    -- 解压固件
    utils.secure_tar_unzip(
        file_path,
        dir,
        0x6400000, -- 最大解压限制100M
        1024
    )
    
    -- 固件升级准备回调
    local ok = chip_callback.upgrade_prepare_func(self, dir)
    if not ok then
        log:error('[RetimerUpgrade] Prepare failed.')
        fw_mgmt:UpdateServiceUpdateServiceProcessReply(ctx.new(), system_id, firmware_type, RET_ERR)
        return
    end

    -- 解析出文件路径对应的cfg配置文件，并获取升级列表
    self.retimer_upgrade_list = self:get_upgrade_list(firmware_chip_type)
    if not self.retimer_upgrade_list then
        log:error('[RetimerUpgrade] Can not find retimer with chiptype %s.', firmware_chip_type)
        fw_mgmt:UpdateServiceUpdateServiceProcessReply(ctx.new(), system_id, firmware_type, RET_ERR)
        return
    end

    local ret = RET_OK
    -- 遍历升级列表，匹配二进制文件
    for _, retimer in pairs(self.retimer_upgrade_list) do
        local ok = chip_callback.upgrade_single_retimer_func(retimer, dir)
        if not ok then
            log:error('[RetimerUpgrade] upgrade retimer %s failed.', retimer.name)
            ret = RET_ERR
        end
    end
    chip_callback.upgrade_finish_func(self)
    fw_mgmt:UpdateServiceUpdateServiceProcessReply(ctx.new(), system_id, firmware_type, ret)
end

-- 通过升级文件路径获取升级固件列表
function retimer_upg_service:get_upgrade_list(firmware_chip_type)
    local retimer_upgrade_list = {}
    for _, retimer in pairs(self.retimer_collection) do
        if firmware_chip_type == retimer:get_chip_type() then
            table.insert(retimer_upgrade_list, retimer)
        end
    end
    return retimer_upgrade_list
end

function retimer_upg_service:on_upgrade_prepare(context, system_id, firmware_type, cfg_path)
    log:notice("[retimer_upg_service] prepare_upgrade, firmware_type:%s", firmware_type)
    local ok, rsp = fw_mgmt:PFileFileChown(ctx.new(), nil, cfg_path, 104, 104)  --设置配置文件属主为secbox
    if not ok then
        log:error('[RetimerUpgrade] chown cfg file failed, error %s', rsp)
        fw_mgmt:UpdateServiceUpdateServicePrepareReply(ctx.new(), system_id, firmware_type, '', RET_ERR)
        return
    end
    self.cfgs = parser_cfg.get_cfgs(cfg_path)
    log:info('[RetimerUpgrade] Prepare retimer upgrade, firmware_type= %s', firmware_type)

    local version = ''
    fw_mgmt:UpdateServiceUpdateServicePrepareReply(ctx.new(), system_id, firmware_type, version, RET_OK)
end

function retimer_upg_service:on_upgrade_process(context, system_id, firmware_type, file_path)
    if not next(self.cfgs) then
        return
    end
    log:notice('[retimer_upg_service] Process retimer upgrade, firmware_type= %s', firmware_type)

    local ok, rsp = fw_mgmt:PFileFileChown(ctx.new(), nil, file_path, 104, 104)
    if not ok then
        log:error('[RetimerUpgrade] chown file failed, error %s', rsp)
        fw_mgmt:UpdateServiceUpdateServiceProcessReply(ctx.new(), system_id, firmware_type, RET_ERR)
        return
    end

    cmn.skynet.fork(function ()
        self:upgrade_task(system_id, firmware_type, file_path)
    end)
end

function retimer_upg_service:on_upgrade_finish(context, system_id, firmware_type)
    if not next(self.cfgs) then
        return
    end
    log:notice('[retimer_upg_service] Finish retimer upgrade, firmware_type= %s', firmware_type)

    fw_mgmt:UpdateServiceUpdateServiceFinishReply(ctx.new(), system_id, firmware_type, RET_OK)
end

function retimer_upg_service.new()
    return setmetatable({}, retimer_upg_service)
end

function retimer_upg_service:init(bus, retimer_collection)
    log:info("[RetimerUpgrade] init")
    self.bus = bus
    self.retimer_collection = retimer_collection
    self.retimer_upgrade_list = {}

    self.cfgs = {}

    self.register_chips_callback = {
        [RETIMER_CONSTANTS.CHIP_TYPE.TYPE_5902L] = require 'retimer.upgrade.retimer_upg_5902l'
    }

    -- 注册固件升级监测
    upgrade_subject.get_instance():register_upgrade_observer(self, 'Retimer')
end

return singleton(retimer_upg_service)