-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

-- Description: 升级服务
local singleton = require 'mc.singleton'
local log = require 'mc.logging'
local fw_mgmt = require 'general_hardware.client'
local defs = require 'mcu.upgrade.defs'
local context = require 'mc.context'
local upgrade_service_object = require 'mcu.upgrade.upgrade_service.upgrade_service_object'
local upgrade_subject = require 'upgrade_subject.upgrade_subject'

local upgrade_service_handle = {}
upgrade_service_handle.__index = upgrade_service_handle

function upgrade_service_handle.new()
    return setmetatable({}, upgrade_service_handle)
end

local function is_firmware_upgrading()
    local firmware_type  = {'Mcu', 'Vrd', 'Vdm'}
    for i = 1, #firmware_type do
        local upgrade_service_obj = upgrade_service_object.get_instance():get_upgrade_service_obj(firmware_type[i])
        if not upgrade_service_obj then
            goto continue
        end
        if upgrade_service_obj:is_firmware_upgrading() then
           return true
        end
        ::continue::
    end
    return false
end

function upgrade_service_handle:is_upgrading_or_activing()
    if self.is_activing then
        log:debug('find firmware is activing')
        return true
    end
    if is_firmware_upgrading() then
        log:debug('find firmware is upgrading')
        return true
    end
    return false
end

function upgrade_service_handle:set_dft_mode()
    self.dft_mode = true
end

function upgrade_service_handle:on_upgrade_prepare(ctx, system_id, firmware_type, cfg_path, hpm_path)
    log:notice('on upgrade prepare start, firmware_type = %s', firmware_type)
    local upgrade_service_obj = upgrade_service_object.get_instance():get_upgrade_service_obj(firmware_type)
    if not upgrade_service_obj then
        log:error('not find upgrade service_obj obj')
        fw_mgmt:UpdateServiceUpdateServicePrepareReply(context.new(), system_id, firmware_type, '', defs.RET.ERR)
        return
    end

    local ok, err = pcall(function ()
        upgrade_service_obj:on_upgrade_prepare(system_id, firmware_type, cfg_path, hpm_path)
    end)
    if not ok then
        log:error('on upgrade prepare failed, err: %s', err)
        fw_mgmt:UpdateServiceUpdateServicePrepareReply(context.new(), system_id, firmware_type, '', defs.RET.ERR)
    end
end

function upgrade_service_handle:on_upgrade_process(ctx, system_id, firmware_type, file_path)
    log:notice('on upgrade process start, firmware_type = %s', firmware_type)
    local upgrade_service_obj = upgrade_service_object.get_instance():get_upgrade_service_obj(firmware_type)
    if not upgrade_service_obj then
        log:error('not find upgrade service_obj obj')
        fw_mgmt:UpdateServiceUpdateServiceProcessReply(context.new(), system_id, firmware_type, defs.RET.ERR)
        return
    end

    local ok, err = pcall(function ()
        upgrade_service_obj:on_upgrade_process(system_id, firmware_type, file_path)
    end)
    if not ok then
        log:error('on upgrade process failed, err: %s', err)
        fw_mgmt:UpdateServiceUpdateServiceProcessReply(context.new(), system_id, firmware_type, '', defs.RET.ERR)
    end
end

function upgrade_service_handle:on_upgrade_finish(ctx, system_id, firmware_type)
    log:notice('on upgrade finish start, firmware_type = %s', firmware_type)
    local upgrade_service_obj = upgrade_service_object.get_instance():get_upgrade_service_obj(firmware_type)
    if not upgrade_service_obj then
        log:error('not find upgrade service_obj obj')
        fw_mgmt:UpdateServiceUpdateServiceFinishReply(context.new(), system_id, firmware_type, defs.RET.ERR)
        return
    end

    local ok, err = pcall(function ()
        upgrade_service_obj:on_upgrade_finish(system_id, firmware_type)
    end)
    if not ok then
        log:error('on upgrade finish failed, err: %s', err)
        fw_mgmt:UpdateServiceUpdateServiceFinishReply(context.new(), system_id, firmware_type, '', defs.RET.ERR)
    end
end

function upgrade_service_handle:on_active_process(ctx, system_id, firmware_type)
    -- 如果存在VRD生效、MCU或者VRD正在升级，则退出不响应，避免导致正在生效的任务提前被终止
    if self:is_upgrading_or_activing() then
        log:notice('[Vrd] is_activing = %s or another MCU or VRD is Upgrading', self.is_activing)
        fw_mgmt:FirmwareActiveFirmwareActiveActiveProcessReply(context.new(), system_id, firmware_type, defs.RET.ERR)
        return
    end

    local upgrade_service_obj = upgrade_service_object.get_instance():get_upgrade_service_obj(defs.firmware_type.VRD)
    if not upgrade_service_obj then
        log:error('not find upgrade service_obj obj')
        fw_mgmt:FirmwareActiveFirmwareActiveActiveProcessReply(context.new(), system_id, firmware_type, defs.RET.ERR)
        return
    end

    self.is_activing = true
    local ok, err = pcall(function ()
        upgrade_service_obj:on_active_process(system_id, firmware_type)
    end)
    if not ok then
        log:error('on active process failed, err: %s', err)
        fw_mgmt:FirmwareActiveFirmwareActiveActiveProcessReply(context.new(), system_id, firmware_type, defs.RET.ERR)
    end
    self.is_activing = false
end

local function mcu_register_observer(self)
    local mcu_firmware_type = {"Mcu", "Vrd", "Vdm"}
    -- 注册固件升级监测
    for i = 1, #mcu_firmware_type do
        upgrade_subject.get_instance():register_upgrade_observer(self, mcu_firmware_type[i])
    end

    -- 注册固件生效监测
    upgrade_subject.get_instance():register_active_observer(self, "VRD")
end

function upgrade_service_handle:init(bus, db, mcu_collection)
    log:notice("upgrade service handle init")
    self.is_activing = false
    self.dft_mode = false

    -- 初始化升级服务对象
    upgrade_service_object.get_instance():init_upgrade_service_obj(bus, db, mcu_collection)

    -- 注册固件监测
    mcu_register_observer(self)
end

return singleton(upgrade_service_handle)