-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

-- Description: fructl的相关工具
local log = require 'mc.logging'
local mdb = require 'mc.mdb'
local context = require 'mc.context'
local skynet = require 'skynet'
local fructl_handler = {}
require 'general_hardware.json_types.Multihost'

local FRUCTRL_ITF = 'bmc.kepler.Systems.FruCtrl'
local POWER_ON_LOCK_ITF = 'bmc.kepler.Systems.PowerOnLock'
--多host整机上下电接口
local CHASSIS_FRUCTRL_ITF = 'bmc.kepler.Chassis.FruCtrl'
local CHASSIS_POWER_ON_LOCK_ITF = 'bmc.kepler.Chassis.PowerOnLock'
local MULTIHOST_ITF <const> = 'bmc.kepler.Managers.Multihost'
local MULTIHOST_PATH <const> = '/bmc/kepler/Managers/1/Multihost'   -- ManagerId为1
local MAX_WAIT_HOSTTYPE_COUNTS<const> = 3

function fructl_handler.get_mdb_sub_path_obj(bus, path, interface)
    local ok, obj_list = pcall(mdb.get_sub_objects, bus, path, interface)
    if not ok or next(obj_list) == nil then
        return nil
    end

    for _, obj in pairs(obj_list) do  -- 当前接口名唯一，故直接返回obj_list的第一个值
        return obj
    end
end

local function get_multihost_obj(bus)
    local cnt = 0
    while cnt < MAX_WAIT_HOSTTYPE_COUNTS do
        local obj = fructl_handler.get_mdb_sub_path_obj(bus, MULTIHOST_PATH, MULTIHOST_ITF)
        if obj then
            return obj
        end
        cnt = cnt + 1
        skynet.sleep(100)
    end
    return nil
end

function fructl_handler.is_multihost_type(bus)
    local obj = get_multihost_obj(bus)
    if not obj then
        return
    end

    log:notice('get_host_type: get host type %s', obj.HostType)
    return obj.HostType == 'Multihost'
end

function fructl_handler.get_fructl_obj(bus, system_id)
    local path = '/bmc/kepler/Systems/' .. system_id .. '/FruCtrl'
    local interface = FRUCTRL_ITF
    local obj = fructl_handler.get_mdb_sub_path_obj(bus, path, interface)
    return obj
end

-- 从资源树上获取电源状态
-- fructl需要有system_id对应的资源树，否则返回nil
function fructl_handler.get_power_status(bus, system_id)
    local obj = fructl_handler.get_fructl_obj(bus, system_id)
    if not obj then
        log:error('get_power_status: get system_id(%s) power object failed', system_id)
        return nil
    end
    log:notice('get_power_state: system[%s] get power power %s', system_id, obj.PowerState)
    return obj.PowerState
end

function fructl_handler.get_chassis_fructl_obj(bus, chassis_id)
    local path = '/bmc/kepler/Chassis/' .. chassis_id .. '/FruCtrl'
    local interface = CHASSIS_FRUCTRL_ITF
    local obj = fructl_handler.get_mdb_sub_path_obj(bus, path, interface)
    return obj
end

-- 从资源树上获取整机电源状态
function fructl_handler.get_chassis_power_status(bus, chassis_id)
    local obj = fructl_handler.get_chassis_fructl_obj(bus, chassis_id)
    if not obj then
        log:error('get_chassis_power_status: get chassis power object failed')
        return nil
    end
    log:notice('get_chassis_power_status: get chassis power %s', obj.PowerState)
    return obj.PowerState
end

function fructl_handler.set_chassis_power_on_lock(bus, chassis_id, state, timeout, reason, app_name)
    local path = '/bmc/kepler/Chassis/' .. chassis_id .. '/FruCtrl'
    local obj = fructl_handler.get_mdb_sub_path_obj(bus, path, CHASSIS_POWER_ON_LOCK_ITF)
    if not obj then
        log:error('set_power_on_lock: get chassis power object failed')
        return nil
    end
    return obj.pcall:SetPowerOnLock(context.new(), state, timeout, app_name, reason)
end
 
function fructl_handler.set_power_on_lock(bus, system_id, state, timeout, reason, app_name)
    local path = '/bmc/kepler/Systems/' .. system_id .. '/FruCtrl'
    local interface = POWER_ON_LOCK_ITF
    local obj = fructl_handler.get_mdb_sub_path_obj(bus, path, interface)
    if not obj then
        log:error('set_power_on_lock: get power object failed')
        return nil
    end
    return obj.pcall:SetPowerOnLock(context.new(), state, timeout, app_name, reason)
end

function fructl_handler.set_poweron_lock_until_success(bus, sys_id, state, time, reason)
    local res
    local cnt = 0
    repeat
        res = fructl_handler.set_power_on_lock(bus, sys_id, state, time, reason, 'general_hardware')
        skynet.sleep(50)
        cnt = cnt + 1
    until res or cnt > 20
    if cnt > 20 and not res then
        log:notice('set power on lock failed, lock state : %s', state and 'lock' or 'unlock')
        return
    end
    log:notice('set power on lock success, lock state : %s', state and 'lock' or 'unlock')
end

function fructl_handler.set_power_state(bus, system_id, state, reason)
    local obj = fructl_handler.get_fructl_obj(bus, system_id)
    if not obj then
        log:error('set_power_state: get power object failed')
        return nil
    end
    return obj.pcall:PowerCtrl(context.new(), state, reason)
end

-- 判断通电开机策略例外场景
function fructl_handler.set_poweron_strategy_exceptions(bus, system_id)
    local EXCEPTION = 1
    local reason, exec, period, priority, ok, err
    local obj = fructl_handler.get_fructl_obj(bus, system_id)
    if not obj then
        log:error('get_poweron_strategy_exceptions: get power object failed')
        return
    end
    -- 下次AC不走通电开机策略
    if obj.PowerOnStrategyExceptions == EXCEPTION then
        reason, exec, period, priority = 'Redfish', 'No', 'Once', 1
        ok, err = pcall(obj.SetPowerOnStrategyExceptions, obj, context.new(), reason, exec, period, priority)
    else
        -- 下次复位需要走通电开机策略
        reason, exec, period, priority = 'ActiveCpld', 'Yes', 'Once', 2
        ok, err = pcall(obj.SetPowerOnStrategyExceptions, obj, context.new(), reason, exec, period, priority)
    end
    if not ok then
        log:error('set power restore fail, err: %s', err)
        return
    end
    log:notice('set power restore successfully, reason: %s, execute: %s, period: %s, priority: %s',
            reason, exec, period, priority)
end

return fructl_handler