-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.


local log = require 'mc.logging'
local m_enums = require 'types.enums'

local customize_cfg = {}

-- 装备定制化默认配置表
local config_default_map = {
    ['BMCSet_POWERRESTOR'] = 'poweron',
    ['BMCSet_PowerButtonLockStatus'] = 0,
    ['BMCSet_PowerRestoreDelayMode'] = 'DefaultDelay',
    ['BMCSet_PowerRestoreDelayTime'] = 0,
}

local poweron_strategy_map = {
    [m_enums.PowerRestorePolicy.AlwaysPowerOff] = 'poweroff',
    [m_enums.PowerRestorePolicy.AlwaysPowerOn] = 'poweron',
    [m_enums.PowerRestorePolicy.LastState] = 'powerrestore',
}
local panel_button_map = {
    [false] = 0,
    [true] = 1,
}

local function record_customize_operation_log(ctx, msg)
    if ctx and not ctx:is_empty() then
        log:operation(ctx:get_initiator(), 'fructrl', msg)
    end
end

local function map_change(map, val)
    local str
    for k, v in pairs(map) do
        if v == val then
            str = k
            break
        end
    end
    return str
end

local function set_poweron_strategy(fructrl, val)
    local str = map_change(poweron_strategy_map, val)
    fructrl:set_PowerOnStrategy(str)
end

local function set_panel_pwrbtn_shield_enabled(button_evt, val)
    local str = map_change(panel_button_map, val)
    button_evt:set_PowerButtonShieldEnabled(str)
end

local function set_poweron_delay_mode(fructrl, val)
    fructrl:set_PowerOnDelayMode(val)
end

local function set_poweron_delay_seconds(fructrl, val)
    -- 传入的单位是100ms,资源树上单位是秒,需要除10
    fructrl:set_PowerOnDelaySeconds(math.floor(val) / 10)
end

local import_callback_map  = {
    ['BMCSet_POWERRESTOR'] = set_poweron_strategy,
    ['BMCSet_PowerButtonLockStatus'] = set_panel_pwrbtn_shield_enabled,
    ['BMCSet_PowerRestoreDelayMode'] = set_poweron_delay_mode,
    ['BMCSet_PowerRestoreDelayTime'] = set_poweron_delay_seconds,
}

-- 配置导入接口
function customize_cfg.on_import(fructrl, button_evt, ctx, customize_data)
    -- 更新装备定制化默认配置表，合法项传入更新设置值
    for item, v in pairs(customize_data) do
        if not import_callback_map [item] then
            log:error('Import %s has no object to process.', item)
        else
            config_default_map[item] = v.Value
        end
    end

    -- 执行所有定制化，空白定制化执行默认值
    for k, fun in pairs(import_callback_map) do
        local ok, ret = pcall(function()
            if k == 'BMCSet_PowerButtonLockStatus' then
                return fun(button_evt, config_default_map[k])
            end
            return fun(fructrl, config_default_map[k])
        end)
        if not ok then
            record_customize_operation_log(ctx, string.format('Set %s through customization failed', k))
            log:error('Import %s has error occured (%s) in customize config.', k, ret)
        else
            record_customize_operation_log(ctx, string.format('Set %s through customization successfully', k))
        end
    end
end

-- 配置导出接口
function customize_cfg.on_export(fructrl, button_evt, ctx)
    local data = {}
    data.BMCSet_POWERRESTOR = poweron_strategy_map[fructrl:get_PowerOnStrategy()]
    log:notice("Export BMCSet_POWERRESTOR %s successfully", data.BMCSet_POWERRESTOR)
    data.BMCSet_PowerButtonLockStatus = panel_button_map[button_evt:get_PowerButtonShieldEnabled()]
    log:notice("Export BMCSet_PowerButtonLockStatus %s successfully", data.BMCSet_PowerButtonLockStatus)
    data.BMCSet_PowerRestoreDelayMode = fructrl:get_PowerOnDelayMode()
    log:notice("Export BMCSet_PowerRestoreDelayMode %s successfully", data.BMCSet_PowerRestoreDelayMode)
    -- 导出的单位是100ms,资源树上单位是秒,需要乘10
    data.BMCSet_PowerRestoreDelayTime = fructrl:get_PowerOnDelaySeconds() * 10
    log:notice("Export BMCSet_PowerRestoreDelayTime %s successfully", data.BMCSet_PowerRestoreDelayTime)
    log:notice('Export customize config successfully')
    return data
end

return customize_cfg
