-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--          http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.
local mdb = require 'mc.mdb'
local class = require 'mc.class'
local app_base = require 'mc.client_app_base'
local mdb_service = require 'mc.mdb.mdb_service'
local subscribe_signal = require 'mc.mdb.subscribe_signal'
local org_freedesktop_dbus = require 'sd_bus.org_freedesktop_dbus'

local match_rule = org_freedesktop_dbus.MatchRule
local get_non_virtual_interface_objects =
    mdb_service.get_non_virtual_interface_objects
local foreach_non_virtual_interface_objects =
    mdb_service.foreach_non_virtual_interface_objects

local UpdateService = require 'bios.json_types.UpdateService'
local FirmwareInventory = require 'bios.json_types.FirmwareInventory'
local FirmwareInfo = require 'bios.json_types.FirmwareInfo'
local FruCtrl = require 'bios.json_types.FruCtrl'
local DelayedPowerOnPolicy = require 'bios.json_types.DelayedPowerOnPolicy'
local PowerOnLock = require 'bios.json_types.PowerOnLock'
local FirmwareVerification = require 'bios.json_types.FirmwareVerification'
local Accessor = require 'bios.json_types.Accessor'
local Watchdog2Status = require 'bios.json_types.Watchdog2Status'
local Events = require 'bios.json_types.Events'
local Memory = require 'bios.json_types.Memory'
local AddrInfo = require 'bios.json_types.AddrInfo'
local FirmwareActive = require 'bios.json_types.FirmwareActive'
local Config = require 'bios.json_types.Config'
local Ums = require 'bios.json_types.Ums'
local Sms = require 'bios.json_types.Sms'
local SmsStatus = require 'bios.json_types.SmsStatus'
local Multihost = require 'bios.json_types.Multihost'
local Component = require 'bios.json_types.Component'
local File = require 'bios.json_types.File'
local Driver = require 'bios.json_types.Driver'

---@class bios_client: BasicClient
local bios_client = class(app_base.Client)

function bios_client:GetDelayedPowerOnPolicyObjects()
    return get_non_virtual_interface_objects(self:get_bus(),
                                             'bmc.kepler.Systems.DelayedPowerOnPolicy',
                                             true)
end

function bios_client:ForeachDelayedPowerOnPolicyObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(),
                                                 'bmc.kepler.Systems.DelayedPowerOnPolicy',
                                                 cb, true)
end

function bios_client:GetPowerOnLockObjects()
    return get_non_virtual_interface_objects(self:get_bus(),
                                             'bmc.kepler.Systems.PowerOnLock',
                                             true)
end

function bios_client:ForeachPowerOnLockObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(),
                                                 'bmc.kepler.Systems.PowerOnLock',
                                                 cb, true)
end

function bios_client:GetFirmwareVerificationObjects()
    return get_non_virtual_interface_objects(self:get_bus(),
                                             'bmc.kepler.Managers.SOC.SecureCore.FirmwareVerification',
                                             true)
end

function bios_client:ForeachFirmwareVerificationObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(),
                                                 'bmc.kepler.Managers.SOC.SecureCore.FirmwareVerification',
                                                 cb, true)
end

function bios_client:GetWatchdog2StatusObjects()
    return get_non_virtual_interface_objects(self:get_bus(),
                                             'bmc.kepler.Watchdog2.Status', true)
end

function bios_client:ForeachWatchdog2StatusObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(),
                                                 'bmc.kepler.Watchdog2.Status',
                                                 cb, true)
end

function bios_client:GetMemoryObjects()
    return get_non_virtual_interface_objects(self:get_bus(),
                                             'bmc.kepler.Systems.Memory', true)
end

function bios_client:ForeachMemoryObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(),
                                                 'bmc.kepler.Systems.Memory',
                                                 cb, true)
end

function bios_client:GetConfigObjects()
    return get_non_virtual_interface_objects(self:get_bus(),
                                             'bmc.kepler.Managers.TPCM.Config',
                                             true)
end

function bios_client:ForeachConfigObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(),
                                                 'bmc.kepler.Managers.TPCM.Config',
                                                 cb, true)
end

function bios_client:GetUmsObjects()
    return get_non_virtual_interface_objects(self:get_bus(),
                                             'bmc.kepler.Managers.UsbDevice.Ums',
                                             true)
end

function bios_client:ForeachUmsObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(),
                                                 'bmc.kepler.Managers.UsbDevice.Ums',
                                                 cb, true)
end

function bios_client:GetSmsObjects()
    return get_non_virtual_interface_objects(self:get_bus(),
                                             'bmc.kepler.Systems.Sms', true)
end

function bios_client:ForeachSmsObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(),
                                                 'bmc.kepler.Systems.Sms', cb,
                                                 true)
end

function bios_client:GetSmsStatusObjects()
    return get_non_virtual_interface_objects(self:get_bus(),
                                             'bmc.kepler.Systems.Sms.SmsStatus',
                                             true)
end

function bios_client:ForeachSmsStatusObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(),
                                                 'bmc.kepler.Systems.Sms.SmsStatus',
                                                 cb, true)
end

function bios_client:GetMultihostObjects()
    return get_non_virtual_interface_objects(self:get_bus(),
                                             'bmc.kepler.Managers.Multihost',
                                             true)
end

function bios_client:ForeachMultihostObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(),
                                                 'bmc.kepler.Managers.Multihost',
                                                 cb, true)
end

function bios_client:GetComponentObjects()
    return get_non_virtual_interface_objects(self:get_bus(),
                                             'bmc.kepler.Systems.Component',
                                             true)
end

function bios_client:ForeachComponentObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(),
                                                 'bmc.kepler.Systems.Component',
                                                 cb, true)
end

function bios_client:GetUpdateServiceUpdateServiceObject()
    return mdb.try_get_object(self:get_bus(), '/bmc/kepler/UpdateService',
                              'bmc.kepler.UpdateService')
end

function bios_client:OnUpdateServiceInterfacesAdded(cb)
    local path_namespace = '/bmc/kepler/UpdateService'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
                                             'bmc.kepler.UpdateService')
end

function bios_client:OnUpdateServiceInterfacesRemoved(cb)
    local path_namespace = '/bmc/kepler/UpdateService'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
                                               cb, 'bmc.kepler.UpdateService')
end

function bios_client:GetFirmwareInventoryFirmwareInventoryObject()
    return mdb.try_get_object(self:get_bus(),
                              '/bmc/kepler/UpdateService/FirmwareInventory',
                              'bmc.kepler.UpdateService.FirmwareInventory')
end

function bios_client:OnFirmwareInventoryInterfacesAdded(cb)
    local path_namespace = '/bmc/kepler/UpdateService/FirmwareInventory'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
                                             'bmc.kepler.UpdateService.FirmwareInventory')
end

function bios_client:OnFirmwareInventoryInterfacesRemoved(cb)
    local path_namespace = '/bmc/kepler/UpdateService/FirmwareInventory'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
                                               cb,
                                               'bmc.kepler.UpdateService.FirmwareInventory')
end

function bios_client:GetFirmwareInfoFirmwareInfoObject(path_params)
    return mdb.try_get_object(self:get_bus(), path_params and
                                  ('/bmc/kepler/UpdateService/FirmwareInventory/' ..
                                      path_params['Id']) or
                                  '/bmc/kepler/UpdateService/FirmwareInventory/:Id',
                              'bmc.kepler.UpdateService.FirmwareInfo')
end

function bios_client:OnFirmwareInfoInterfacesAdded(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/UpdateService/FirmwareInventory/' ..
                                   path_params['Id']) or
                               '/bmc/kepler/UpdateService/FirmwareInventory'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
                                             'bmc.kepler.UpdateService.FirmwareInfo')
end

function bios_client:OnFirmwareInfoInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/UpdateService/FirmwareInventory/' ..
                                   path_params['Id']) or
                               '/bmc/kepler/UpdateService/FirmwareInventory'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
                                               cb,
                                               'bmc.kepler.UpdateService.FirmwareInfo')
end

function bios_client:GetFruCtrlFruCtrlObject(path_params)
    return mdb.try_get_object(self:get_bus(), path_params and
                                  ('/bmc/kepler/Systems/' ..
                                      path_params['SystemId'] .. '/FruCtrl/' ..
                                      path_params['Id']) or
                                  '/bmc/kepler/Systems/:SystemId/FruCtrl/:Id',
                              'bmc.kepler.Systems.FruCtrl')
end

function bios_client:OnFruCtrlPropertiesChanged(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/Systems/' ..
                                   path_params['SystemId'] .. '/FruCtrl/' ..
                                   path_params['Id']) or '/bmc/kepler/Systems'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_properties_changed(self:get_bus(), path_namespace,
                                               cb, 'bmc.kepler.Systems.FruCtrl',
                                               {
            "PowerState", "SysResetDetected"
        })
end

function bios_client:OnFruCtrlInterfacesAdded(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/Systems/' ..
                                   path_params['SystemId'] .. '/FruCtrl/' ..
                                   path_params['Id']) or '/bmc/kepler/Systems'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
                                             'bmc.kepler.Systems.FruCtrl')
end

function bios_client:OnFruCtrlInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/Systems/' ..
                                   path_params['SystemId'] .. '/FruCtrl/' ..
                                   path_params['Id']) or '/bmc/kepler/Systems'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
                                               cb, 'bmc.kepler.Systems.FruCtrl')
end

function bios_client:OnDelayedPowerOnPolicyInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
                                             'bmc.kepler.Systems.DelayedPowerOnPolicy')
end

function bios_client:OnDelayedPowerOnPolicyInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
                                               'bmc.kepler.Systems.DelayedPowerOnPolicy')
end

function bios_client:OnPowerOnLockInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
                                             'bmc.kepler.Systems.PowerOnLock')
end

function bios_client:OnPowerOnLockInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
                                               'bmc.kepler.Systems.PowerOnLock')
end

function bios_client:OnFirmwareVerificationInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
                                             'bmc.kepler.Managers.SOC.SecureCore.FirmwareVerification')
end

function bios_client:OnFirmwareVerificationInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
                                               'bmc.kepler.Managers.SOC.SecureCore.FirmwareVerification')
end

function bios_client:GetAccessorAccessorObject(path_params)
    return mdb.try_get_object(self:get_bus(),
                              path_params and
                                  ('/bmc/kepler/Accessor/' .. path_params['Id']) or
                                  '/bmc/kepler/Accessor/:Id',
                              'bmc.kepler.Accessor')
end

function bios_client:OnAccessorInterfacesAdded(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/Accessor/' .. path_params['Id']) or
                               '/bmc/kepler/Accessor'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
                                             'bmc.kepler.Accessor')
end

function bios_client:OnAccessorInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/Accessor/' .. path_params['Id']) or
                               '/bmc/kepler/Accessor'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
                                               cb, 'bmc.kepler.Accessor')
end

function bios_client:OnWatchdog2StatusInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
                                             'bmc.kepler.Watchdog2.Status')
end

function bios_client:OnWatchdog2StatusInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
                                               'bmc.kepler.Watchdog2.Status')
end

function bios_client:GetEventsEventsObject(path_params)
    return mdb.try_get_object(self:get_bus(), path_params and
                                  ('/bmc/kepler/Systems/' ..
                                      path_params['SystemId'] .. '/Events') or
                                  '/bmc/kepler/Systems/:SystemId/Events',
                              'bmc.kepler.Systems.Events')
end

function bios_client:OnEventsInterfacesAdded(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/Systems/' ..
                                   path_params['SystemId'] .. '/Events') or
                               '/bmc/kepler/Systems'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
                                             'bmc.kepler.Systems.Events')
end

function bios_client:OnEventsInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/Systems/' ..
                                   path_params['SystemId'] .. '/Events') or
                               '/bmc/kepler/Systems'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
                                               cb, 'bmc.kepler.Systems.Events')
end

function bios_client:OnMemoryInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
                                             'bmc.kepler.Systems.Memory')
end

function bios_client:OnMemoryInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
                                               'bmc.kepler.Systems.Memory')
end

function bios_client:GetDriveAddrInfoObject(path_params)
    return mdb.try_get_object(self:get_bus(), path_params and
                                  ('/bmc/kepler/Systems/' ..
                                      path_params['SystemId'] ..
                                      '/Storage/Drives/' .. path_params['Id']) or
                                  '/bmc/kepler/Systems/:SystemId/Storage/Drives/:Id',
                              'bmc.kepler.Systems.Storage.Drive.AddrInfo')
end

function bios_client:OnAddrInfoInterfacesAdded(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/Systems/' ..
                                   path_params['SystemId'] .. '/Storage/Drives/' ..
                                   path_params['Id']) or '/bmc/kepler/Systems'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
                                             'bmc.kepler.Systems.Storage.Drive.AddrInfo')
end

function bios_client:OnAddrInfoInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/Systems/' ..
                                   path_params['SystemId'] .. '/Storage/Drives/' ..
                                   path_params['Id']) or '/bmc/kepler/Systems'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
                                               cb,
                                               'bmc.kepler.Systems.Storage.Drive.AddrInfo')
end

function bios_client:GetFirmwareActiveFirmwareActiveObject()
    return mdb.try_get_object(self:get_bus(),
                              '/bmc/kepler/UpdateService/FirmwareActive',
                              'bmc.kepler.UpdateService.FirmwareActive')
end

function bios_client:OnFirmwareActiveInterfacesAdded(cb)
    local path_namespace = '/bmc/kepler/UpdateService/FirmwareActive'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
                                             'bmc.kepler.UpdateService.FirmwareActive')
end

function bios_client:OnFirmwareActiveInterfacesRemoved(cb)
    local path_namespace = '/bmc/kepler/UpdateService/FirmwareActive'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
                                               cb,
                                               'bmc.kepler.UpdateService.FirmwareActive')
end

function bios_client:OnConfigInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
                                             'bmc.kepler.Managers.TPCM.Config')
end

function bios_client:OnConfigInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
                                               'bmc.kepler.Managers.TPCM.Config')
end

function bios_client:OnUmsInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
                                             'bmc.kepler.Managers.UsbDevice.Ums')
end

function bios_client:OnUmsInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
                                               'bmc.kepler.Managers.UsbDevice.Ums')
end

function bios_client:OnSmsInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
                                             'bmc.kepler.Systems.Sms')
end

function bios_client:OnSmsInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
                                               'bmc.kepler.Systems.Sms')
end

function bios_client:OnSmsStatusInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
                                             'bmc.kepler.Systems.Sms.SmsStatus')
end

function bios_client:OnSmsStatusInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
                                               'bmc.kepler.Systems.Sms.SmsStatus')
end

function bios_client:OnMultihostInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
                                             'bmc.kepler.Managers.Multihost')
end

function bios_client:OnMultihostInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
                                               'bmc.kepler.Managers.Multihost')
end

function bios_client:OnComponentInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
                                             'bmc.kepler.Systems.Component')
end

function bios_client:OnComponentInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
                                               'bmc.kepler.Systems.Component')
end

function bios_client:GetFileFileObject(path_params)
    return mdb.try_get_object(self:get_bus(), path_params and
                                  ('/bmc/kepler/Managers/' .. path_params['Id'] ..
                                      '/Security/File') or
                                  '/bmc/kepler/Managers/:Id/Security/File',
                              'bmc.kepler.Managers.Security.File')
end

function bios_client:OnFileInterfacesAdded(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/Managers/' .. path_params['Id'] ..
                                   '/Security/File') or '/bmc/kepler/Managers'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
                                             'bmc.kepler.Managers.Security.File')
end

function bios_client:OnFileInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/Managers/' .. path_params['Id'] ..
                                   '/Security/File') or '/bmc/kepler/Managers'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
                                               cb,
                                               'bmc.kepler.Managers.Security.File')
end

function bios_client:GetProxyDriverObject(path_params)
    return mdb.try_get_object(self:get_bus(), path_params and
                                  ('/bmc/kepler/Managers/' .. path_params['Id'] ..
                                      '/Security/Proxy') or
                                  '/bmc/kepler/Managers/:Id/Security/Proxy',
                              'bmc.kepler.Managers.Security.Proxy.Driver')
end

function bios_client:OnDriverInterfacesAdded(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/Managers/' .. path_params['Id'] ..
                                   '/Security/Proxy') or '/bmc/kepler/Managers'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
                                             'bmc.kepler.Managers.Security.Proxy.Driver')
end

function bios_client:OnDriverInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and
                               ('/bmc/kepler/Managers/' .. path_params['Id'] ..
                                   '/Security/Proxy') or '/bmc/kepler/Managers'
    self.signal_slots[#self.signal_slots + 1] =
        subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
                                               cb,
                                               'bmc.kepler.Managers.Security.Proxy.Driver')
end

---@param FilePath string
---@param Parameters UpdateService.Object
---@return UpdateService.StartUpgradeRsp
function bios_client:UpdateServiceUpdateServiceStartUpgrade(ctx, FilePath,
                                                            Parameters)
    local req =
        UpdateService.StartUpgradeReq.new(FilePath, Parameters):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.StartUpgradeRsp.new(
               obj:StartUpgrade(ctx, req:unpack(true)))
end

function bios_client:PUpdateServiceUpdateServiceStartUpgrade(ctx, FilePath,
                                                             Parameters)
    return pcall(function()
        local req =
            UpdateService.StartUpgradeReq.new(FilePath, Parameters):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.StartUpgradeRsp.new(
                   obj:StartUpgrade(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@param FirmwareType string
---@param VersionStr string
---@param PrepareResult integer
---@return UpdateService.PrepareReplyRsp
function bios_client:UpdateServiceUpdateServicePrepareReply(ctx, SystemId,
                                                            FirmwareType,
                                                            VersionStr,
                                                            PrepareResult)
    local req = UpdateService.PrepareReplyReq.new(SystemId, FirmwareType,
                                                  VersionStr, PrepareResult):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.PrepareReplyRsp.new(
               obj:PrepareReply(ctx, req:unpack(true)))
end

function bios_client:PUpdateServiceUpdateServicePrepareReply(ctx, SystemId,
                                                             FirmwareType,
                                                             VersionStr,
                                                             PrepareResult)
    return pcall(function()
        local req = UpdateService.PrepareReplyReq.new(SystemId, FirmwareType,
                                                      VersionStr, PrepareResult):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.PrepareReplyRsp.new(
                   obj:PrepareReply(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@param FirmwareType string
---@param ProcessResult integer
---@return UpdateService.ProcessReplyRsp
function bios_client:UpdateServiceUpdateServiceProcessReply(ctx, SystemId,
                                                            FirmwareType,
                                                            ProcessResult)
    local req = UpdateService.ProcessReplyReq.new(SystemId, FirmwareType,
                                                  ProcessResult):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.ProcessReplyRsp.new(
               obj:ProcessReply(ctx, req:unpack(true)))
end

function bios_client:PUpdateServiceUpdateServiceProcessReply(ctx, SystemId,
                                                             FirmwareType,
                                                             ProcessResult)
    return pcall(function()
        local req = UpdateService.ProcessReplyReq.new(SystemId, FirmwareType,
                                                      ProcessResult):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.ProcessReplyRsp.new(
                   obj:ProcessReply(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@param FirmwareType string
---@param FinishResult integer
---@return UpdateService.FinishReplyRsp
function bios_client:UpdateServiceUpdateServiceFinishReply(ctx, SystemId,
                                                           FirmwareType,
                                                           FinishResult)
    local req = UpdateService.FinishReplyReq.new(SystemId, FirmwareType,
                                                 FinishResult):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.FinishReplyRsp.new(
               obj:FinishReply(ctx, req:unpack(true)))
end

function bios_client:PUpdateServiceUpdateServiceFinishReply(ctx, SystemId,
                                                            FirmwareType,
                                                            FinishResult)
    return pcall(function()
        local req = UpdateService.FinishReplyReq.new(SystemId, FirmwareType,
                                                     FinishResult):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.FinishReplyRsp.new(
                   obj:FinishReply(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@param FirmwareType string
---@param ResultCode integer
---@param Progress integer
---@param ResultDetail string
---@return UpdateService.UpdateUpgradeStatusRsp
function bios_client:UpdateServiceUpdateServiceUpdateUpgradeStatus(ctx,
                                                                   SystemId,
                                                                   FirmwareType,
                                                                   ResultCode,
                                                                   Progress,
                                                                   ResultDetail)
    local req = UpdateService.UpdateUpgradeStatusReq.new(SystemId, FirmwareType,
                                                         ResultCode, Progress,
                                                         ResultDetail):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.UpdateUpgradeStatusRsp.new(
               obj:UpdateUpgradeStatus(ctx, req:unpack(true)))
end

function bios_client:PUpdateServiceUpdateServiceUpdateUpgradeStatus(ctx,
                                                                    SystemId,
                                                                    FirmwareType,
                                                                    ResultCode,
                                                                    Progress,
                                                                    ResultDetail)
    return pcall(function()
        local req = UpdateService.UpdateUpgradeStatusReq.new(SystemId,
                                                             FirmwareType,
                                                             ResultCode,
                                                             Progress,
                                                             ResultDetail):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.UpdateUpgradeStatusRsp.new(
                   obj:UpdateUpgradeStatus(ctx, req:unpack(true)))
    end)
end

---@param TaskId integer
---@return UpdateService.GetUpgradeStatusRsp
function bios_client:UpdateServiceUpdateServiceGetUpgradeStatus(ctx, TaskId)
    local req = UpdateService.GetUpgradeStatusReq.new(TaskId):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.GetUpgradeStatusRsp.new(
               obj:GetUpgradeStatus(ctx, req:unpack(true)))
end

function bios_client:PUpdateServiceUpdateServiceGetUpgradeStatus(ctx, TaskId)
    return pcall(function()
        local req = UpdateService.GetUpgradeStatusReq.new(TaskId):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.GetUpgradeStatusRsp.new(
                   obj:GetUpgradeStatus(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@param Id string
---@return UpdateService.RollbackRsp
function bios_client:UpdateServiceUpdateServiceRollback(ctx, SystemId, Id)
    local req = UpdateService.RollbackReq.new(SystemId, Id):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.RollbackRsp.new(obj:Rollback(ctx, req:unpack(true)))
end

function bios_client:PUpdateServiceUpdateServiceRollback(ctx, SystemId, Id)
    return pcall(function()
        local req = UpdateService.RollbackReq.new(SystemId, Id):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.RollbackRsp
                   .new(obj:Rollback(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@param Id string
---@return UpdateService.ExportRsp
function bios_client:UpdateServiceUpdateServiceExport(ctx, SystemId, Id)
    local req = UpdateService.ExportReq.new(SystemId, Id):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.ExportRsp.new(obj:Export(ctx, req:unpack(true)))
end

function bios_client:PUpdateServiceUpdateServiceExport(ctx, SystemId, Id)
    return pcall(function()
        local req = UpdateService.ExportReq.new(SystemId, Id):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.ExportRsp.new(obj:Export(ctx, req:unpack(true)))
    end)
end

---@param FilePath string
---@param FirmwareType string
---@return UpdateService.CheckFirmwareTypeRsp
function bios_client:UpdateServiceUpdateServiceCheckFirmwareType(ctx, FilePath,
                                                                 FirmwareType)
    local req =
        UpdateService.CheckFirmwareTypeReq.new(FilePath, FirmwareType):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.CheckFirmwareTypeRsp.new(
               obj:CheckFirmwareType(ctx, req:unpack(true)))
end

function bios_client:PUpdateServiceUpdateServiceCheckFirmwareType(ctx, FilePath,
                                                                  FirmwareType)
    return pcall(function()
        local req = UpdateService.CheckFirmwareTypeReq.new(FilePath,
                                                           FirmwareType):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.CheckFirmwareTypeRsp.new(
                   obj:CheckFirmwareType(ctx, req:unpack(true)))
    end)
end

---@param FilePath string
---@return UpdateService.ParseFirmwarePackageRsp
function bios_client:UpdateServiceUpdateServiceParseFirmwarePackage(ctx,
                                                                    FilePath)
    local req = UpdateService.ParseFirmwarePackageReq.new(FilePath):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.ParseFirmwarePackageRsp.new(
               obj:ParseFirmwarePackage(ctx, req:unpack(true)))
end

function bios_client:PUpdateServiceUpdateServiceParseFirmwarePackage(ctx,
                                                                     FilePath)
    return pcall(function()
        local req =
            UpdateService.ParseFirmwarePackageReq.new(FilePath):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.ParseFirmwarePackageRsp.new(
                   obj:ParseFirmwarePackage(ctx, req:unpack(true)))
    end)
end

---@param FirmwareInfoTable FirmwareInventory.Object
---@param Updateable boolean
---@param Capability integer
---@param MaxPackageSizeMiB integer
---@return FirmwareInventory.AddRsp
function bios_client:FirmwareInventoryFirmwareInventoryAdd(ctx,
                                                           FirmwareInfoTable,
                                                           Updateable,
                                                           Capability,
                                                           MaxPackageSizeMiB)
    local req = FirmwareInventory.AddReq.new(FirmwareInfoTable, Updateable,
                                             Capability, MaxPackageSizeMiB):validate()
    local obj = self:GetFirmwareInventoryFirmwareInventoryObject()

    return FirmwareInventory.AddRsp.new(obj:Add(ctx, req:unpack(true)))
end

function bios_client:PFirmwareInventoryFirmwareInventoryAdd(ctx,
                                                            FirmwareInfoTable,
                                                            Updateable,
                                                            Capability,
                                                            MaxPackageSizeMiB)
    return pcall(function()
        local req = FirmwareInventory.AddReq.new(FirmwareInfoTable, Updateable,
                                                 Capability, MaxPackageSizeMiB):validate()
        local obj = self:GetFirmwareInventoryFirmwareInventoryObject()

        return FirmwareInventory.AddRsp.new(obj:Add(ctx, req:unpack(true)))
    end)
end

---@param Id string
---@return FirmwareInventory.DeleteRsp
function bios_client:FirmwareInventoryFirmwareInventoryDelete(ctx, Id)
    local req = FirmwareInventory.DeleteReq.new(Id):validate()
    local obj = self:GetFirmwareInventoryFirmwareInventoryObject()

    return FirmwareInventory.DeleteRsp.new(obj:Delete(ctx, req:unpack(true)))
end

function bios_client:PFirmwareInventoryFirmwareInventoryDelete(ctx, Id)
    return pcall(function()
        local req = FirmwareInventory.DeleteReq.new(Id):validate()
        local obj = self:GetFirmwareInventoryFirmwareInventoryObject()

        return FirmwareInventory.DeleteRsp
                   .new(obj:Delete(ctx, req:unpack(true)))
    end)
end

---@param PowerCtrlType string
---@param RestartCause string
---@return FruCtrl.PowerCtrlRsp
function bios_client:FruCtrlFruCtrlPowerCtrl(ctx, path_params, PowerCtrlType,
                                             RestartCause)
    local req = FruCtrl.PowerCtrlReq.new(PowerCtrlType, RestartCause):validate()
    local obj = self:GetFruCtrlFruCtrlObject(path_params)

    return FruCtrl.PowerCtrlRsp.new(obj:PowerCtrl(ctx, req:unpack(true)))
end

function bios_client:PFruCtrlFruCtrlPowerCtrl(ctx, path_params, PowerCtrlType,
                                              RestartCause)
    return pcall(function()
        local req =
            FruCtrl.PowerCtrlReq.new(PowerCtrlType, RestartCause):validate()
        local obj = self:GetFruCtrlFruCtrlObject(path_params)

        return FruCtrl.PowerCtrlRsp.new(obj:PowerCtrl(ctx, req:unpack(true)))
    end)
end

---@param Reason string
---@param Execute string
---@param EffectivePeriod string
---@param Priority integer
---@return FruCtrl.SetPowerOnStrategyExceptionsRsp
function bios_client:FruCtrlFruCtrlSetPowerOnStrategyExceptions(ctx,
                                                                path_params,
                                                                Reason, Execute,
                                                                EffectivePeriod,
                                                                Priority)
    local req = FruCtrl.SetPowerOnStrategyExceptionsReq.new(Reason, Execute,
                                                            EffectivePeriod,
                                                            Priority):validate()
    local obj = self:GetFruCtrlFruCtrlObject(path_params)

    return FruCtrl.SetPowerOnStrategyExceptionsRsp.new(
               obj:SetPowerOnStrategyExceptions(ctx, req:unpack(true)))
end

function bios_client:PFruCtrlFruCtrlSetPowerOnStrategyExceptions(ctx,
                                                                 path_params,
                                                                 Reason,
                                                                 Execute,
                                                                 EffectivePeriod,
                                                                 Priority)
    return pcall(function()
        local req = FruCtrl.SetPowerOnStrategyExceptionsReq.new(Reason, Execute,
                                                                EffectivePeriod,
                                                                Priority):validate()
        local obj = self:GetFruCtrlFruCtrlObject(path_params)

        return FruCtrl.SetPowerOnStrategyExceptionsRsp.new(
                   obj:SetPowerOnStrategyExceptions(ctx, req:unpack(true)))
    end)
end

---@param Type integer
---@return FruCtrl.SetACLostRsp
function bios_client:FruCtrlFruCtrlSetACLost(ctx, path_params, Type)
    local req = FruCtrl.SetACLostReq.new(Type):validate()
    local obj = self:GetFruCtrlFruCtrlObject(path_params)

    return FruCtrl.SetACLostRsp.new(obj:SetACLost(ctx, req:unpack(true)))
end

function bios_client:PFruCtrlFruCtrlSetACLost(ctx, path_params, Type)
    return pcall(function()
        local req = FruCtrl.SetACLostReq.new(Type):validate()
        local obj = self:GetFruCtrlFruCtrlObject(path_params)

        return FruCtrl.SetACLostRsp.new(obj:SetACLost(ctx, req:unpack(true)))
    end)
end

---@param PowerCtrlType string
---@param RestartCause string
---@param FruID integer
---@return FruCtrl.FruPowerCtrlRsp
function bios_client:FruCtrlFruCtrlFruPowerCtrl(ctx, path_params, PowerCtrlType,
                                                RestartCause, FruID)
    local req =
        FruCtrl.FruPowerCtrlReq.new(PowerCtrlType, RestartCause, FruID):validate()
    local obj = self:GetFruCtrlFruCtrlObject(path_params)

    return FruCtrl.FruPowerCtrlRsp.new(obj:FruPowerCtrl(ctx, req:unpack(true)))
end

function bios_client:PFruCtrlFruCtrlFruPowerCtrl(ctx, path_params,
                                                 PowerCtrlType, RestartCause,
                                                 FruID)
    return pcall(function()
        local req = FruCtrl.FruPowerCtrlReq.new(PowerCtrlType, RestartCause,
                                                FruID):validate()
        local obj = self:GetFruCtrlFruCtrlObject(path_params)

        return FruCtrl.FruPowerCtrlRsp.new(
                   obj:FruPowerCtrl(ctx, req:unpack(true)))
    end)
end

---@param EventCode string
---@param Enabled integer
---@param ObjectIndex integer
---@param MockState integer
---@return Events.MockEventRsp
function bios_client:EventsEventsMockEvent(ctx, path_params, EventCode, Enabled,
                                           ObjectIndex, MockState)
    local req = Events.MockEventReq.new(EventCode, Enabled, ObjectIndex,
                                        MockState):validate()
    local obj = self:GetEventsEventsObject(path_params)

    return Events.MockEventRsp.new(obj:MockEvent(ctx, req:unpack(true)))
end

function bios_client:PEventsEventsMockEvent(ctx, path_params, EventCode,
                                            Enabled, ObjectIndex, MockState)
    return pcall(function()
        local req = Events.MockEventReq.new(EventCode, Enabled, ObjectIndex,
                                            MockState):validate()
        local obj = self:GetEventsEventsObject(path_params)

        return Events.MockEventRsp.new(obj:MockEvent(ctx, req:unpack(true)))
    end)
end

---@param StartId integer
---@param Count integer
---@param QueryParameters Events.KeyValueTable[]
---@return Events.GetAlarmListRsp
function bios_client:EventsEventsGetAlarmList(ctx, path_params, StartId, Count,
                                              QueryParameters)
    local req =
        Events.GetAlarmListReq.new(StartId, Count, QueryParameters):validate()
    local obj = self:GetEventsEventsObject(path_params)

    return Events.GetAlarmListRsp.new(obj:GetAlarmList(ctx, req:unpack(true)))
end

function bios_client:PEventsEventsGetAlarmList(ctx, path_params, StartId, Count,
                                               QueryParameters)
    return pcall(function()
        local req =
            Events.GetAlarmListReq.new(StartId, Count, QueryParameters):validate()
        local obj = self:GetEventsEventsObject(path_params)

        return Events.GetAlarmListRsp.new(
                   obj:GetAlarmList(ctx, req:unpack(true)))
    end)
end

---@param StartId integer
---@param Count integer
---@param QueryParameters Events.KeyValueTable[]
---@return Events.GetEventListRsp
function bios_client:EventsEventsGetEventList(ctx, path_params, StartId, Count,
                                              QueryParameters)
    local req =
        Events.GetEventListReq.new(StartId, Count, QueryParameters):validate()
    local obj = self:GetEventsEventsObject(path_params)

    return Events.GetEventListRsp.new(obj:GetEventList(ctx, req:unpack(true)))
end

function bios_client:PEventsEventsGetEventList(ctx, path_params, StartId, Count,
                                               QueryParameters)
    return pcall(function()
        local req =
            Events.GetEventListReq.new(StartId, Count, QueryParameters):validate()
        local obj = self:GetEventsEventsObject(path_params)

        return Events.GetEventListRsp.new(
                   obj:GetEventList(ctx, req:unpack(true)))
    end)
end

---@return Events.ClearEventListRsp
function bios_client:EventsEventsClearEventList(ctx, path_params)
    local obj = self:GetEventsEventsObject(path_params)

    return Events.ClearEventListRsp.new(obj:ClearEventList(ctx))
end

function bios_client:PEventsEventsClearEventList(ctx, path_params)
    return pcall(function()
        local obj = self:GetEventsEventsObject(path_params)

        return Events.ClearEventListRsp.new(obj:ClearEventList(ctx))
    end)
end

---@return Events.GetEventInfoRsp
function bios_client:EventsEventsGetEventInfo(ctx, path_params)
    local obj = self:GetEventsEventsObject(path_params)

    return Events.GetEventInfoRsp.new(obj:GetEventInfo(ctx))
end

function bios_client:PEventsEventsGetEventInfo(ctx, path_params)
    return pcall(function()
        local obj = self:GetEventsEventsObject(path_params)

        return Events.GetEventInfoRsp.new(obj:GetEventInfo(ctx))
    end)
end

---@param EventCode string
---@param MaskState integer
---@param Mode integer
---@return Events.MaskEventRsp
function bios_client:EventsEventsMaskEvent(ctx, path_params, EventCode,
                                           MaskState, Mode)
    local req = Events.MaskEventReq.new(EventCode, MaskState, Mode):validate()
    local obj = self:GetEventsEventsObject(path_params)

    return Events.MaskEventRsp.new(obj:MaskEvent(ctx, req:unpack(true)))
end

function bios_client:PEventsEventsMaskEvent(ctx, path_params, EventCode,
                                            MaskState, Mode)
    return pcall(function()
        local req =
            Events.MaskEventReq.new(EventCode, MaskState, Mode):validate()
        local obj = self:GetEventsEventsObject(path_params)

        return Events.MaskEventRsp.new(obj:MaskEvent(ctx, req:unpack(true)))
    end)
end

---@return Events.GetMaskedEventListRsp
function bios_client:EventsEventsGetMaskedEventList(ctx, path_params)
    local obj = self:GetEventsEventsObject(path_params)

    return Events.GetMaskedEventListRsp.new(obj:GetMaskedEventList(ctx))
end

function bios_client:PEventsEventsGetMaskedEventList(ctx, path_params)
    return pcall(function()
        local obj = self:GetEventsEventsObject(path_params)

        return Events.GetMaskedEventListRsp.new(obj:GetMaskedEventList(ctx))
    end)
end

---@param EventInfo Events.KeyValueTable[]
---@return Events.AddEventRsp
function bios_client:EventsEventsAddEvent(ctx, path_params, EventInfo)
    local req = Events.AddEventReq.new(EventInfo):validate()
    local obj = self:GetEventsEventsObject(path_params)

    return Events.AddEventRsp.new(obj:AddEvent(ctx, req:unpack(true)))
end

function bios_client:PEventsEventsAddEvent(ctx, path_params, EventInfo)
    return pcall(function()
        local req = Events.AddEventReq.new(EventInfo):validate()
        local obj = self:GetEventsEventsObject(path_params)

        return Events.AddEventRsp.new(obj:AddEvent(ctx, req:unpack(true)))
    end)
end

---@param Path string
---@return Events.ExportEventRsp
function bios_client:EventsEventsExportEvent(ctx, path_params, Path)
    local req = Events.ExportEventReq.new(Path):validate()
    local obj = self:GetEventsEventsObject(path_params)

    return Events.ExportEventRsp.new(obj:ExportEvent(ctx, req:unpack(true)))
end

function bios_client:PEventsEventsExportEvent(ctx, path_params, Path)
    return pcall(function()
        local req = Events.ExportEventReq.new(Path):validate()
        local obj = self:GetEventsEventsObject(path_params)

        return Events.ExportEventRsp.new(obj:ExportEvent(ctx, req:unpack(true)))
    end)
end

---@param EventName string[]
---@return Events.CheckEventNameRsp
function bios_client:EventsEventsCheckEventName(ctx, path_params, EventName)
    local req = Events.CheckEventNameReq.new(EventName):validate()
    local obj = self:GetEventsEventsObject(path_params)

    return Events.CheckEventNameRsp.new(
               obj:CheckEventName(ctx, req:unpack(true)))
end

function bios_client:PEventsEventsCheckEventName(ctx, path_params, EventName)
    return pcall(function()
        local req = Events.CheckEventNameReq.new(EventName):validate()
        local obj = self:GetEventsEventsObject(path_params)

        return Events.CheckEventNameRsp.new(
                   obj:CheckEventName(ctx, req:unpack(true)))
    end)
end

---@param AlarmName string
---@return Events.SetAlarmNameRsp
function bios_client:EventsEventsSetAlarmName(ctx, path_params, AlarmName)
    local req = Events.SetAlarmNameReq.new(AlarmName):validate()
    local obj = self:GetEventsEventsObject(path_params)

    return Events.SetAlarmNameRsp.new(obj:SetAlarmName(ctx, req:unpack(true)))
end

function bios_client:PEventsEventsSetAlarmName(ctx, path_params, AlarmName)
    return pcall(function()
        local req = Events.SetAlarmNameReq.new(AlarmName):validate()
        local obj = self:GetEventsEventsObject(path_params)

        return Events.SetAlarmNameRsp.new(
                   obj:SetAlarmName(ctx, req:unpack(true)))
    end)
end

---@param EventCode string
---@param Severity integer
---@return Events.SetEventSeverityRsp
function bios_client:EventsEventsSetEventSeverity(ctx, path_params, EventCode,
                                                  Severity)
    local req = Events.SetEventSeverityReq.new(EventCode, Severity):validate()
    local obj = self:GetEventsEventsObject(path_params)

    return Events.SetEventSeverityRsp.new(
               obj:SetEventSeverity(ctx, req:unpack(true)))
end

function bios_client:PEventsEventsSetEventSeverity(ctx, path_params, EventCode,
                                                   Severity)
    return pcall(function()
        local req =
            Events.SetEventSeverityReq.new(EventCode, Severity):validate()
        local obj = self:GetEventsEventsObject(path_params)

        return Events.SetEventSeverityRsp.new(
                   obj:SetEventSeverity(ctx, req:unpack(true)))
    end)
end

---@param EventKeyId string
---@param Action integer
---@return Events.SetEventActionRsp
function bios_client:EventsEventsSetEventAction(ctx, path_params, EventKeyId,
                                                Action)
    local req = Events.SetEventActionReq.new(EventKeyId, Action):validate()
    local obj = self:GetEventsEventsObject(path_params)

    return Events.SetEventActionRsp.new(
               obj:SetEventAction(ctx, req:unpack(true)))
end

function bios_client:PEventsEventsSetEventAction(ctx, path_params, EventKeyId,
                                                 Action)
    return pcall(function()
        local req = Events.SetEventActionReq.new(EventKeyId, Action):validate()
        local obj = self:GetEventsEventsObject(path_params)

        return Events.SetEventActionRsp.new(
                   obj:SetEventAction(ctx, req:unpack(true)))
    end)
end

---@param EventInfo Events.KeyValueTable[]
---@return Events.AddSelRsp
function bios_client:EventsEventsAddSel(ctx, path_params, EventInfo)
    local req = Events.AddSelReq.new(EventInfo):validate()
    local obj = self:GetEventsEventsObject(path_params)

    return Events.AddSelRsp.new(obj:AddSel(ctx, req:unpack(true)))
end

function bios_client:PEventsEventsAddSel(ctx, path_params, EventInfo)
    return pcall(function()
        local req = Events.AddSelReq.new(EventInfo):validate()
        local obj = self:GetEventsEventsObject(path_params)

        return Events.AddSelRsp.new(obj:AddSel(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@return FirmwareActive.StartActiveRsp
function bios_client:FirmwareActiveFirmwareActiveStartActive(ctx, SystemId)
    local req = FirmwareActive.StartActiveReq.new(SystemId):validate()
    local obj = self:GetFirmwareActiveFirmwareActiveObject()

    return FirmwareActive.StartActiveRsp.new(
               obj:StartActive(ctx, req:unpack(true)))
end

function bios_client:PFirmwareActiveFirmwareActiveStartActive(ctx, SystemId)
    return pcall(function()
        local req = FirmwareActive.StartActiveReq.new(SystemId):validate()
        local obj = self:GetFirmwareActiveFirmwareActiveObject()

        return FirmwareActive.StartActiveRsp.new(
                   obj:StartActive(ctx, req:unpack(true)))
    end)
end

---@param ActiveAction FirmwareActive.KeyValueTable[]
---@return FirmwareActive.RegisterActiveActionRsp
function bios_client:FirmwareActiveFirmwareActiveRegisterActiveAction(ctx,
                                                                      ActiveAction)
    local req =
        FirmwareActive.RegisterActiveActionReq.new(ActiveAction):validate()
    local obj = self:GetFirmwareActiveFirmwareActiveObject()

    return FirmwareActive.RegisterActiveActionRsp.new(
               obj:RegisterActiveAction(ctx, req:unpack(true)))
end

function bios_client:PFirmwareActiveFirmwareActiveRegisterActiveAction(ctx,
                                                                       ActiveAction)
    return pcall(function()
        local req =
            FirmwareActive.RegisterActiveActionReq.new(ActiveAction):validate()
        local obj = self:GetFirmwareActiveFirmwareActiveObject()

        return FirmwareActive.RegisterActiveActionRsp.new(
                   obj:RegisterActiveAction(ctx, req:unpack(true)))
    end)
end

---@param ActiveStatus FirmwareActive.KeyValueTable[]
---@return FirmwareActive.UpdateActiveStatusRsp
function bios_client:FirmwareActiveFirmwareActiveUpdateActiveStatus(ctx,
                                                                    ActiveStatus)
    local req =
        FirmwareActive.UpdateActiveStatusReq.new(ActiveStatus):validate()
    local obj = self:GetFirmwareActiveFirmwareActiveObject()

    return FirmwareActive.UpdateActiveStatusRsp.new(
               obj:UpdateActiveStatus(ctx, req:unpack(true)))
end

function bios_client:PFirmwareActiveFirmwareActiveUpdateActiveStatus(ctx,
                                                                     ActiveStatus)
    return pcall(function()
        local req =
            FirmwareActive.UpdateActiveStatusReq.new(ActiveStatus):validate()
        local obj = self:GetFirmwareActiveFirmwareActiveObject()

        return FirmwareActive.UpdateActiveStatusRsp.new(
                   obj:UpdateActiveStatus(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@param FirmwareType string
---@param ProcessResult integer
---@return FirmwareActive.ActiveProcessReplyRsp
function bios_client:FirmwareActiveFirmwareActiveActiveProcessReply(ctx,
                                                                    SystemId,
                                                                    FirmwareType,
                                                                    ProcessResult)
    local req = FirmwareActive.ActiveProcessReplyReq.new(SystemId, FirmwareType,
                                                         ProcessResult):validate()
    local obj = self:GetFirmwareActiveFirmwareActiveObject()

    return FirmwareActive.ActiveProcessReplyRsp.new(
               obj:ActiveProcessReply(ctx, req:unpack(true)))
end

function bios_client:PFirmwareActiveFirmwareActiveActiveProcessReply(ctx,
                                                                     SystemId,
                                                                     FirmwareType,
                                                                     ProcessResult)
    return pcall(function()
        local req = FirmwareActive.ActiveProcessReplyReq.new(SystemId,
                                                             FirmwareType,
                                                             ProcessResult):validate()
        local obj = self:GetFirmwareActiveFirmwareActiveObject()

        return FirmwareActive.ActiveProcessReplyRsp.new(
                   obj:ActiveProcessReply(ctx, req:unpack(true)))
    end)
end

---@param SrcFile string
---@param DstFile string
---@param Uid integer
---@param Gid integer
---@return File.CopyRsp
function bios_client:FileFileCopy(ctx, path_params, SrcFile, DstFile, Uid, Gid)
    local req = File.CopyReq.new(SrcFile, DstFile, Uid, Gid):validate()
    local obj = self:GetFileFileObject(path_params)

    return File.CopyRsp.new(obj:Copy(ctx, req:unpack(true)))
end

function bios_client:PFileFileCopy(ctx, path_params, SrcFile, DstFile, Uid, Gid)
    return pcall(function()
        local req = File.CopyReq.new(SrcFile, DstFile, Uid, Gid):validate()
        local obj = self:GetFileFileObject(path_params)

        return File.CopyRsp.new(obj:Copy(ctx, req:unpack(true)))
    end)
end

---@param SrcFile string
---@param DstFile string
---@param Uid integer
---@param Gid integer
---@return File.MoveRsp
function bios_client:FileFileMove(ctx, path_params, SrcFile, DstFile, Uid, Gid)
    local req = File.MoveReq.new(SrcFile, DstFile, Uid, Gid):validate()
    local obj = self:GetFileFileObject(path_params)

    return File.MoveRsp.new(obj:Move(ctx, req:unpack(true)))
end

function bios_client:PFileFileMove(ctx, path_params, SrcFile, DstFile, Uid, Gid)
    return pcall(function()
        local req = File.MoveReq.new(SrcFile, DstFile, Uid, Gid):validate()
        local obj = self:GetFileFileObject(path_params)

        return File.MoveRsp.new(obj:Move(ctx, req:unpack(true)))
    end)
end

---@param DstFile string
---@param FileMode integer
---@return File.ChmodRsp
function bios_client:FileFileChmod(ctx, path_params, DstFile, FileMode)
    local req = File.ChmodReq.new(DstFile, FileMode):validate()
    local obj = self:GetFileFileObject(path_params)

    return File.ChmodRsp.new(obj:Chmod(ctx, req:unpack(true)))
end

function bios_client:PFileFileChmod(ctx, path_params, DstFile, FileMode)
    return pcall(function()
        local req = File.ChmodReq.new(DstFile, FileMode):validate()
        local obj = self:GetFileFileObject(path_params)

        return File.ChmodRsp.new(obj:Chmod(ctx, req:unpack(true)))
    end)
end

---@param DstFile string
---@param Uid integer
---@param Gid integer
---@return File.ChownRsp
function bios_client:FileFileChown(ctx, path_params, DstFile, Uid, Gid)
    local req = File.ChownReq.new(DstFile, Uid, Gid):validate()
    local obj = self:GetFileFileObject(path_params)

    return File.ChownRsp.new(obj:Chown(ctx, req:unpack(true)))
end

function bios_client:PFileFileChown(ctx, path_params, DstFile, Uid, Gid)
    return pcall(function()
        local req = File.ChownReq.new(DstFile, Uid, Gid):validate()
        local obj = self:GetFileFileObject(path_params)

        return File.ChownRsp.new(obj:Chown(ctx, req:unpack(true)))
    end)
end

---@param DstFile string
---@return File.DeleteRsp
function bios_client:FileFileDelete(ctx, path_params, DstFile)
    local req = File.DeleteReq.new(DstFile):validate()
    local obj = self:GetFileFileObject(path_params)

    return File.DeleteRsp.new(obj:Delete(ctx, req:unpack(true)))
end

function bios_client:PFileFileDelete(ctx, path_params, DstFile)
    return pcall(function()
        local req = File.DeleteReq.new(DstFile):validate()
        local obj = self:GetFileFileObject(path_params)

        return File.DeleteRsp.new(obj:Delete(ctx, req:unpack(true)))
    end)
end

---@param DstFile string
---@param OpenMode string
---@param FileMode integer
---@param Uid integer
---@param Gid integer
---@return File.CreateRsp
function bios_client:FileFileCreate(ctx, path_params, DstFile, OpenMode,
                                    FileMode, Uid, Gid)
    local req =
        File.CreateReq.new(DstFile, OpenMode, FileMode, Uid, Gid):validate()
    local obj = self:GetFileFileObject(path_params)

    return File.CreateRsp.new(obj:Create(ctx, req:unpack(true)))
end

function bios_client:PFileFileCreate(ctx, path_params, DstFile, OpenMode,
                                     FileMode, Uid, Gid)
    return pcall(function()
        local req =
            File.CreateReq.new(DstFile, OpenMode, FileMode, Uid, Gid):validate()
        local obj = self:GetFileFileObject(path_params)

        return File.CreateRsp.new(obj:Create(ctx, req:unpack(true)))
    end)
end

---@param DstFile string
---@return File.ChangeOwnerRsp
function bios_client:FileFileChangeOwner(ctx, path_params, DstFile)
    local req = File.ChangeOwnerReq.new(DstFile):validate()
    local obj = self:GetFileFileObject(path_params)

    return File.ChangeOwnerRsp.new(obj:ChangeOwner(ctx, req:unpack(true)))
end

function bios_client:PFileFileChangeOwner(ctx, path_params, DstFile)
    return pcall(function()
        local req = File.ChangeOwnerReq.new(DstFile):validate()
        local obj = self:GetFileFileObject(path_params)

        return File.ChangeOwnerRsp.new(obj:ChangeOwner(ctx, req:unpack(true)))
    end)
end

---@param DstFile string
---@param Permission string
---@return File.IsPermittedRsp
function bios_client:FileFileIsPermitted(ctx, path_params, DstFile, Permission)
    local req = File.IsPermittedReq.new(DstFile, Permission):validate()
    local obj = self:GetFileFileObject(path_params)

    return File.IsPermittedRsp.new(obj:IsPermitted(ctx, req:unpack(true)))
end

function bios_client:PFileFileIsPermitted(ctx, path_params, DstFile, Permission)
    return pcall(function()
        local req = File.IsPermittedReq.new(DstFile, Permission):validate()
        local obj = self:GetFileFileObject(path_params)

        return File.IsPermittedRsp.new(obj:IsPermitted(ctx, req:unpack(true)))
    end)
end

---@param ModulePath string
---@param ExtendParams Driver.ExtendParams
---@return Driver.InsmodRsp
function bios_client:ProxyDriverInsmod(ctx, path_params, ModulePath,
                                       ExtendParams)
    local req = Driver.InsmodReq.new(ModulePath, ExtendParams):validate()
    local obj = self:GetProxyDriverObject(path_params)

    return Driver.InsmodRsp.new(obj:Insmod(ctx, req:unpack(true)))
end

function bios_client:PProxyDriverInsmod(ctx, path_params, ModulePath,
                                        ExtendParams)
    return pcall(function()
        local req = Driver.InsmodReq.new(ModulePath, ExtendParams):validate()
        local obj = self:GetProxyDriverObject(path_params)

        return Driver.InsmodRsp.new(obj:Insmod(ctx, req:unpack(true)))
    end)
end

---@param ModulePath string
---@param ExtendParams Driver.ExtendParams
---@return Driver.RmmodRsp
function bios_client:ProxyDriverRmmod(ctx, path_params, ModulePath, ExtendParams)
    local req = Driver.RmmodReq.new(ModulePath, ExtendParams):validate()
    local obj = self:GetProxyDriverObject(path_params)

    return Driver.RmmodRsp.new(obj:Rmmod(ctx, req:unpack(true)))
end

function bios_client:PProxyDriverRmmod(ctx, path_params, ModulePath,
                                       ExtendParams)
    return pcall(function()
        local req = Driver.RmmodReq.new(ModulePath, ExtendParams):validate()
        local obj = self:GetProxyDriverObject(path_params)

        return Driver.RmmodRsp.new(obj:Rmmod(ctx, req:unpack(true)))
    end)
end

function bios_client:SubscribeUpdateServiceUpdateServiceUpgradePrepareSignal(cb)
    local sig = match_rule.signal('UpgradePrepareSignal',
                                  'bmc.kepler.UpdateService'):with_path(
                    "/bmc/kepler/UpdateService")
    self.signal_slots[#self.signal_slots + 1] =
        self:get_bus():match(sig, function(msg) cb(msg:read()) end)
end

function bios_client:SubscribeUpdateServiceUpdateServiceUpgradeProcessSignal(cb)
    local sig = match_rule.signal('UpgradeProcessSignal',
                                  'bmc.kepler.UpdateService'):with_path(
                    "/bmc/kepler/UpdateService")
    self.signal_slots[#self.signal_slots + 1] =
        self:get_bus():match(sig, function(msg) cb(msg:read()) end)
end

function bios_client:SubscribeUpdateServiceUpdateServiceUpgradeFinishSignal(cb)
    local sig = match_rule.signal('UpgradeFinishSignal',
                                  'bmc.kepler.UpdateService'):with_path(
                    "/bmc/kepler/UpdateService")
    self.signal_slots[#self.signal_slots + 1] =
        self:get_bus():match(sig, function(msg) cb(msg:read()) end)
end

function bios_client:SubscribeUpdateServiceUpdateServiceRollbackSignal(cb)
    local sig =
        match_rule.signal('RollbackSignal', 'bmc.kepler.UpdateService'):with_path(
            "/bmc/kepler/UpdateService")
    self.signal_slots[#self.signal_slots + 1] =
        self:get_bus():match(sig, function(msg) cb(msg:read()) end)
end

function bios_client:SubscribeUpdateServiceUpdateServiceExportSignal(cb)
    local sig =
        match_rule.signal('ExportSignal', 'bmc.kepler.UpdateService'):with_path(
            "/bmc/kepler/UpdateService")
    self.signal_slots[#self.signal_slots + 1] =
        self:get_bus():match(sig, function(msg) cb(msg:read()) end)
end

function bios_client:SubscribeFruCtrlFruCtrlBeforePowerOnSignal(cb)
    local sig = match_rule.signal('BeforePowerOnSignal',
                                  'bmc.kepler.Systems.FruCtrl'):with_path(
                    "/bmc/kepler/Systems/:SystemId/FruCtrl/:Id")
    self.signal_slots[#self.signal_slots + 1] =
        self:get_bus():match(sig, function(msg) cb(msg:read()) end)
end

function bios_client:SubscribeFirmwareActiveFirmwareActiveActiveProcessSignal(cb)
    local sig = match_rule.signal('ActiveProcessSignal',
                                  'bmc.kepler.UpdateService.FirmwareActive'):with_path(
                    "/bmc/kepler/UpdateService/FirmwareActive")
    self.signal_slots[#self.signal_slots + 1] =
        self:get_bus():match(sig, function(msg) cb(msg:read()) end)
end

function bios_client:ctor() self.signal_slots = {} end

---@type bios_client
return bios_client.new('bios')
