#!/bin/bash
# Package the firmware upgrade scripts in the IPMC and DATA areas.
# Copyright (c) 2024 Huawei Technologies Co., Ltd.
# openUBMC is licensed under Mulan PSL v2.
# You can use this software according to the terms and conditions of the Mulan PSL v2.
# You may obtain a copy of Mulan PSL v2 at:
#         http://license.coscl.org.cn/MulanPSL2
# THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
# EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
# MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
# See the Mulan PSL v2 for more details.

set -e
#*****************************************************************************
# Function     : make_hpm_file
# Description  : Packet the EEPROM file in hpm package.
# Parameter:
#   input: NA
#   output: NA
# Return Value : NA
#
#  History         :
#  1.Date          : 2023/01/05
#    Modification : Created function
#
#*****************************************************************************
function make_hpm_file()
{
    ls *.bin
    if [ $? -ne 0 ]; then
        echo "Binary not exist!"
        exit -1
    fi

    filesizecheck CfgFileList.conf
    tar --format=gnu -czvf conf.tar.gz beforeaction.sh afteraction.sh CfgFileList.conf update.cfg
    if ls *.frud &> /dev/null; then
        tar --format=gnu -czvf eeprom.tar.gz *.bin *.frud
    else
        tar --format=gnu -czvf eeprom.tar.gz *.bin
    fi

    hpmimage hpm_devkit.config
    rm -rf *.gz & rm -rf *.bin

    if [ ! -f "devkit.hpm" ]
    then
        echo "Generate devkit.hpm Failed."
        exit 1
    else
        echo "Generate devkit.hpm Success."
    fi
}

#*****************************************************************************
# Function     : update_image_filelist
# Description  : Update hpm image.filelist.
# Parameter:
#   input: NA
#   output: NA
# Return Value : NA
#
#  History         :
#  1.Date          : 2023/01/05
#    Modification : Created function
#
#*****************************************************************************
function update_image_filelist()
{
    local sha256val=$(sha256sum devkit.hpm | awk '{print $1}')
    cat <<EOF > image.filelist
Manifest Version: 1.0
Create By: openUBMC
Name: image.hpm
SHA256-Digest: ${sha256val}
EOF
}

#*****************************************************************************
# Function     : add_len_to_hpm
# Description  : Calculate the file length and write it to the hpm package.
# Parameter:
#   input: NA
#   output: NA
# Return Value : NA
#
#  History         :
#  1.Date          : 2023/01/05
#    Modification : Created function
#
#****************************************************************************
function add_len_to_hpm()
{
    file_len=$(stat -c%s $1)
    file_len_format=$(printf %08x $file_len)
    if [ 8 != $(echo $file_len_format | wc -L) ]
    then
        echo "$1 length is invalid, $file_len_format"
        exit 1
    fi
    echo -n $file_len_format >> $2
}

#*****************************************************************************
# Function     : rebuild_hpm
# Description  : Rebuild the HPM Package Using the Signature Result Files.
# Parameter:
#   input: NA
#   output: NA
# Return Value : NA
#
#  History         :
#  1.Date          : 2023/01/05
#    Modification : Created function
#
#*****************************************************************************
function rebuild_hpm()
{
    local hpm_file="${HPM_PACKAGE_PATH}/devkit.hpm"
    local hpm_temp="${HPM_PACKAGE_PATH}/devkit_temp.hpm"
    local file_list="${HPM_PACKAGE_PATH}/image.filelist"
    local file_list_cms="${HPM_PACKAGE_PATH}/image.filelist.cms"
    local sign_crl="${HPM_PACKAGE_PATH}/cms.crl"

    #add file num
    echo -n $(printf %08x 3) >> $hpm_temp
    #add file list id and length
    echo -n $(printf %08x 1) >> $hpm_temp
    add_len_to_hpm $file_list $hpm_temp

    #add cms file id and length
    echo -n $(printf %08x 2) >> $hpm_temp
    echo "cms placeholder" > $file_list_cms
    add_len_to_hpm $file_list_cms $hpm_temp

    #add crl file id and length
    echo -n $(printf %08x 3) >> $hpm_temp
    echo "crl placeholder" > $sign_crl
    add_len_to_hpm $sign_crl $hpm_temp

    #rebuild hpm file
    cat $file_list >> $hpm_temp
    cat $file_list_cms >> $hpm_temp
    cat $sign_crl >> $hpm_temp
    cat $hpm_file >> $hpm_temp
    mv $hpm_temp $hpm_file
}

if [[ $1 = "package" ]]; then
    HPM_PACKAGE_PATH=$2
    make_hpm_file
    update_image_filelist
elif [[ $1 = "rebuild" ]]; then
    HPM_PACKAGE_PATH=$2
    rebuild_hpm
fi
