#!/usr/bin/python
# -*- coding: UTF-8 -*-
# Copyright (c) 2024 Huawei Technologies Co., Ltd.
# openUBMC is licensed under Mulan PSL v2.
# You can use this software according to the terms and conditions of the Mulan PSL v2.
# You may obtain a copy of Mulan PSL v2 at:
#         http://license.coscl.org.cn/MulanPSL2
# THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
# EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
# MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
# See the Mulan PSL v2 for more details.

'''
功    能：下载并安装rtos/hcc工具
修改记录：2021-10-11 创建
'''
import os
import shutil

from bmcgo.tasks.task import Task
from bmcgo.utils.config import Config
from bmcgo.tasks.download_buildtools_hm import DownloadHmBuildTools
from bmcgo import misc


class DownloadDefaultBuildtools(Task):
    def __init__(self, config: Config):
        super(DownloadDefaultBuildtools, self).__init__(config, "DownloadDefaultBuildtools")
        self.rtos_sdk_dir = f"{self.config.tools_path}/rtos-sdk-arm64"
        self.dependency_file = os.path.realpath(os.path.join(
            self.config.code_path, self.get_manufacture_config("base/dependency_buildtools")))
        self.dep_md5 = "/opt/RTOS/dependency.md5sum"
        self.dep_md5_new = "{}/dependency.md5sum.new".format(os.environ["HOME"])
        self.skip_install = False

    def download_tools(self):
        self.info(f"移除下载路径: {self.rtos_sdk_dir}")
        self.run_command(f"rm -rf {self.rtos_sdk_dir}", ignore_error=True, sudo=True)
        self.info('开始下载依赖工具...')
        partner_tools_dir = f"{os.path.expanduser('~')}/rtos_compiler"
        if self.config.partner_mode:
            self.info(f"从缓存目录{partner_tools_dir}复制编译器工具")
            self.run_command(f"cp -rf {partner_tools_dir}/. {self.rtos_sdk_dir}")
        self.info("下载依赖工具结束")

    def check_rtos_sdk(self):
        is_ubuntu = self.tools.is_ubuntu
        self.chdir(self.rtos_sdk_dir)
        self.info("安装 rpm 包")
        self.info("删除目录 /opt/RTOS")
        self.run_command("rm -rf /opt/RTOS", sudo=True)
        for rpm in os.listdir("./"):
            if not os.path.isfile(rpm) or not rpm.endswith(".rpm"):
                continue
            self.info("安装 {}".format(rpm))
            if not is_ubuntu:
                self.run_command("rpm -ivh {}".format(rpm), sudo=True)
            else:
                self.pipe_command(["rpm2cpio {}".format(rpm), "sudo cpio -id -D /"])

        self.info("删除目录 /opt/hcc_arm64le")
        self.run_command("rm -rf /opt/hcc_arm64le", sudo=True)
        self.info("解压 hcc_arm64le")
        self.run_command("tar -xzf hcc_arm64le.tar.gz -C /opt", sudo=True)

        self.info("删除目录 /opt/hi1711sdk")
        self.run_command(f"rm -rf /opt/hi1711sdk", ignore_error=True, sudo=True)
        self.run_command(f"mkdir -p /opt/hi1711sdk", ignore_error=True, sudo=True)
        self.info("解压 hi1711sdk")
        self.run_command("tar -xzf hi1711sdk.tar.gz -C /opt/hi1711sdk Module.symvers", sudo=True)

        logname = os.getenv(misc.ENV_LOGNAME, None)
        if logname and logname != "root":
            self.run_command("chown {} /opt/hcc_arm64le -R".format(logname), sudo=True)
            self.run_command("chown {} /opt/RTOS -R".format(logname), sudo=True)
            self.run_command("chown {} /opt/hi1711sdk -R".format(logname), sudo=True)
        self.chdir(self.config.code_path)

        libstdcpp_install_path = f"{self.config.sysroot}/usr/"
        os.makedirs(libstdcpp_install_path, exist_ok=True)
        self.run_command(f"cp -rf {self.config.cross_compile_install_path}/{self.config.cross_prefix}/lib64/" +
                         f" {libstdcpp_install_path}")
        self.run_command("cp -af {} {}".format(self.dep_md5_new, self.dep_md5))
        # md5文件对齐其他分支
        dep_md5_br = "{}/dependency.md5sum".format(os.environ["HOME"])
        self.run_command(f"cp -af {self.dep_md5_new} {dep_md5_br}")

    def run(self):
        self.pipe_command([f"md5sum {self.dependency_file}"], self.dep_md5_new)
        if os.path.isfile(self.dep_md5):
            with open(self.dep_md5, "r") as fp:
                md5old = fp.read()
            with open(self.dep_md5_new, "r") as fp:
                md5new = fp.read()
            if md5old == md5new:
                self.info("版本匹配, 跳过安装")
                self.skip_install = True
                return
        self.download_tools()

    def install(self):
        if self.skip_install:
            return
        # 检查rtos是否安装，未安装或版本不匹配时安装
        self.check_rtos_sdk()


class TaskClass(Task):
    def __init__(self, config, work_name=""):
        super(TaskClass, self).__init__(config, work_name)
        self.prepare_conan()
        profile, _ = self.get_profile_config()
        self.download_buildtools = None
        if profile and profile.settings.get("os") == "HongMeng":
            self.download_buildtools = DownloadHmBuildTools(config)
        else:
            self.download_buildtools = DownloadDefaultBuildtools(config)

    def run(self):
        self.download_buildtools.run()

    def install(self):
        self.download_buildtools.install()